(function ($) {
    var validVerbs = ["POST", "PUT", "DELETE"];

    /**
     * @constructor
     *
     * @param {(jQuery|string|HTMLElement)} form the form element (a jQuery selector or the element itself)
     * @param {string} verb the verb to use for submitting the form (one of: 'POST', 'PUT', 'DELETE')
     * @param {object} [options]
     * @param {function(validation: FormValidation, fields: object)} [options.presubmitValidator] validation callback function
     * @param {Function} [options.dataFormatter]
     * @param {(jQuery|string|HTMLElement)} [options.$throbberContainer]
     * @param {boolean} [options.insertErrorsAutomagically=true] whether to automatically insert errors after a failed submit
     */
    function UMForm(form, verb, options) {
        options = this._validateOptions(options);
        if (!~validVerbs.indexOf(verb.toUpperCase())) {
            throw new TypeError("Supplied verb was not one of " + validVerbs.join(", "));
        }

        /** functions to use for client-side form validation **/
        this.presubmitValidators = [ options.presubmitValidator ];
        /** used to pre-process data before sending it to the server*/
        this.dataFormatter = options.dataFormatter;
        /** whether to automatically insert errors returned by the server */
        this.insertErrorsAutomagically = options.insertErrorsAutomagically;
        /** callbacks for form submit always/done/fail events */
        this.callbacks = {};
        this.$form = $(form).submit($.proxy(this.onSubmit, this, verb));
        /** hash containing fields that have changed since this form was initialised */
        this._changedFields = {};
        this._initChangedFieldsHandler();
        var modalForm = this.$form.closest('.dialog-components');
        if (modalForm.length) {
            //If we are in a modal dialog, the submit button is outside the form. Go hunting!
            this.$submitButton = modalForm.find('button.aui-button-primary');
            this.$throbberContainer = options.$throbberContainer || modalForm.find(".dialog-button-panel");
        } else {
            this.$submitButton = this.$form.find("input, button").filter("[type='submit']");
            this.$throbberContainer = options.$throbberContainer || this.$form.find(".buttons-container .buttons");
        }

        this.$throbber = $(aui.icons.icon({icon: "wait"})).hide();
        this.$throbber.appendTo(this.$throbberContainer);
        this._errors = [];
    }

    /**
     * Adds a validator to this form. Calling this method triggers validation.
     *
     * @param {function(validation: FormValidation, fields: object)} validator validation callback function
     */
    UMForm.prototype.addValidator = function (validator) {
        if (!_.isFunction(validator)) {
            throw new TypeError('Validator is not a function');
        }

        this.presubmitValidators.push(validator);
        this._validateForm(this._formData());
    };

    /**
     * @private
     */
    UMForm.prototype.onSubmit = function (verb, event) {
        event.preventDefault();
        this._setSubmitting(true);

        /** true if the user has tried to submit */
        this._hasTriedSubmit = true;

        var formData = this._formData();
        var valid = this._validateForm(formData);
        if (!valid) {
            this._setSubmitting(false);
            this.focusFirstError(true);
            return;
        }

        $.ajax(this.$form.attr("action"), {
            data: formData.json,
            contentType: "application/json",
            dataType: "json",
            type: verb
        })
            .always($.proxy(this._setSubmitting, this, false))
            .always($.proxy(this._executeCallbacks, this, "always"))
            .done($.proxy(this._removeErrors, this, false))
            .done($.proxy(this._executeCallbacks, this, "done"))
            .fail($.proxy(this._executeCallbacks, this, "fail"))
            .fail($.proxy(function () {
                this.insertErrorsAutomagically && this._insertErrorsAutomagically.apply(this, arguments);
                this.$form.trigger('fail');
            }, this))
        ;
    };

    /**
     * Return JSON representation of the form's input names and values.
     *
     * @returns {{json: string, fields: object}} form data as JSON and as an object
     * @private
     */
    UMForm.prototype._formData = function () {
        var data = {};
        var $form = this.$form;

        // Disabled inputs aren't submitted, by design.
        // However, we disable inputs for "readonly", and
        // we still want them to submit. So, if an input
        // is disabled, let's put them as part of the
        // form's data unless marked as data-submit=false
        var $disabledInputs = $form.find("[disabled]").filter("[data-submit!='false']");
        $disabledInputs.prop("disabled", false);

        $.each($form.serializeArray(), function (key, pair) {
            var fieldName = pair.name;
            var fieldValue = pair.value;
            var checkbox = false;

            if (!!$form.find("input[type='checkbox'][name='" + fieldName + "']").length) {
                //aui uses "on" for checkboxes, convert to true or false for Java
                if (fieldValue === "on") {
                    fieldValue = true;
                }
                checkbox = true;
            }

            if (data[fieldName] === undefined) {
                if (checkbox && fieldValue !== true) {
                    //checkboxs are key -> array<values> unless they are boolean
                    data[fieldName] = [fieldValue];
                } else {
                    //non-checkboxs can just be key -> value
                    data[fieldName] = fieldValue;
                }
            } else {
                //if we see a second form field with the same name, make it an array.
                if (!_.isArray(data[fieldName])) {
                    data[fieldName] = [data[fieldName]];
                }
                data[fieldName].push(fieldValue);
            }
        });

        $disabledInputs.prop("disabled", true);
        return {
            fields: data,
            json: JSON.stringify(this.dataFormatter(data))
        };
    };

    /**
     * @private
     */
    UMForm.prototype._executeCallbacks = function (event) {
        var callbacks = this.callbacks[event] || [];
        var $form = this.$form;
        var args = $.makeArray(arguments).slice(1);
        $.each(callbacks, function (index, callback) {
            callback.apply($form, args);
        });
    };

    /**
     * @param {boolean} [clientSideOnly=true] true to remove client-side errors only
     * @private
     */
    UMForm.prototype._removeErrors = function (clientSideOnly) {
        clientSideOnly = clientSideOnly !== false;

        var errorsToRemove = _(this._errors).chain()
                .filter(clientSideOnly ? function (obj) { return obj.isClientSide; } : function () { return true; })
                .each(function (error) { error.message.remove(); })
                .value();

        this._errors = _.difference(this._errors, errorsToRemove);
    };

    /**
     * Inserts the message after the <b>last</b> input with the field name. The location of the error can be overridden
     * by setting the <code>data-insert-error-after="#my-selector"<code> attribute on the field.
     *
     * @param {string} name a field name
     * @param {string} message a field error message
     * @param {boolean} isClientSide whether the error was generated client-side or not
     * @param {int} fieldIndex The index of field in error when there are multiple fields with the same name.
     * @private
     */
    UMForm.prototype.insertError = function(name, message, isClientSide, fieldIndex) {
        var $error = $(aui.form.fieldError({ message: message }));
        var selector = "input[name=" + name + "]," + "#s2id_" + name + ",[name='" + name + "']";
        $error.data("selector", selector);
        $error.data("fieldIndex", fieldIndex);
        $error.data("client-side", isClientSide);
        this._errors.push({
            name: name,
            message: $error,
            isClientSide: isClientSide
        });

        // insert the error after the *last* input with the field name. this works nicely
        // for multi-valued fields like checkboxes, where we want the error after all the choices.
        var field = fieldIndex === undefined ? this.$form.find(selector).last() : this.$form.find(selector).eq(fieldIndex);

        var insertAfterSelector = field.attr('data-insert-error-after');
        var insertAfter = insertAfterSelector && AJS.$(insertAfterSelector, this.$form);
        if (insertAfter && insertAfter.length == 1) {
            insertAfter.after($error);
        } else {
            //for checkboxes make sure we place the error after the input or label which ever comes last.
            if (field.attr('type') === 'checkbox') {
                field.siblings('label').addBack().last().after($error);
            } else {
                field.after($error);
            }
        }
    };

    /**
     * @param {boolean} isClientSide true to focus on the first client-side error
     * @private
     */
    UMForm.prototype.focusFirstError = function(isClientSide) {
        // Focus the first field with an error.
        // Rather than using relative finding for the
        // input element, we read it out from above.
        // Less prone to breakage if elements move.
        var errors = this.$form.find("div.error");
        var fieldIndex;
        for (var i = 0; i < errors.length; i++) {
            var error = errors.eq(i);

            if (!isClientSide || error.data("client-side") === true) {
                // focus the corresponding input
                fieldIndex = error.data("fieldIndex") || 0;
                $(error.data("selector")).eq(fieldIndex).focus();
                return;
            }
        }
    };

    /**
     * Inserts server-generated errors.
     *
     * @private
     */
    UMForm.prototype._insertErrorsAutomagically = function (jqXhr) {
        var fieldErrors = {},
            genericErrors = [];

        try {
            $.each(JSON.parse(jqXhr.responseText).errors, function (i, error) {
                if (error.path) {
                    // is field error
                    fieldErrors[error.path] = fieldErrors[error.path] || [];
                    fieldErrors[error.path].push({message: error.message, fieldIndex: error.fieldIndex});
                } else {
                    // is generic error
                    genericErrors[genericErrors.length] = error.message;
                }
            });

            // remove all errors before applying server-side errors
            this._removeErrors(false);
            this._applyValidationResult({
                fieldErrors: fieldErrors,
                genericErrors: genericErrors,
                isClientSide: false
            });

            this.focusFirstError(false);

        } catch (e) {}
    };

    /**
     * @param {string} event the event add a callback for (one of "done", "always", "fail")
     * @param {Function} callback
     * @returns {UMForm}
     */
    UMForm.prototype.on = function (event, callback) {
        this.callbacks[event] = this.callbacks[event] || [];
        this.callbacks[event].push(callback);
        return this;
    };

    /**
     * Calls {@link UMForm#presubmitValidator} to validate the form contents, inserting returned errors into the form.
     *
     * @param {{json: string, fields: {all: Array, changed: Array}}} formData the form data
     * @return {boolean} true if the form is valid, false otherwise
     * @private
     */
    UMForm.prototype._validateForm = function(formData) {
        this._removeErrors();

        var form = this;
        var validation = new FormValidation();

        // hit all the validation callbacks
        _.each(this.presubmitValidators, function(validator) {
            validator.call(form, validation, formData.fields);
        });

        // apply validation results
        this._applyValidationResult(validation);
        var hasErrors = validation.hasErrors();

        // trigger the appropriate events on the form
        this.$form.trigger(hasErrors ? 'fail' : 'pass');
        return !hasErrors;
    };

    /**
     * @param {(object|FormValidation)} validationResult a validation result returned by the server or a pre-submit validator
     * @param {object} validationResult.fieldErrors errors specific to a particular field
     * @param {Array} validationResult.genericErrors errors that aren't specific to a particular field
     * @param {boolean} [validationResult.isClientSide] whether the validation was client-side or not
     * @private
     */
    UMForm.prototype._applyValidationResult = function(validationResult) {
        validationResult = _.defaults(validationResult || {}, {
            isClientSide: true
        });

        var fieldErrors = validationResult.fieldErrors;
        if (!_.isEmpty(fieldErrors)) {
            /**
             * fields for which we will show errors. to prevent showing a wall of errors while the form is being edited
             * we only show errors for fields that the user has changed since loading the form. once the user tries to
             * submit we start showing all errors.
             */
            var showErrorFields;
            if (!this._hasTriedSubmit) {
                showErrorFields = _.pick.apply(_, [fieldErrors].concat(_.keys(this._changedFields)));
            } else {
                showErrorFields = fieldErrors;
            }

            // display each error under the respective field
            _.each(showErrorFields, _.bind(function(messageArray, name) {
                _.each(messageArray, _.bind(function(messageObject){
                    var message;
                    var fieldIndex;
                    if (_.isObject(messageObject)) {
                        message = messageObject.message;
                        fieldIndex = messageObject.fieldIndex;
                    } else {
                        message = messageObject;
                    }
                    this.insertError(name, message, validationResult.isClientSide, fieldIndex);
                }, this));
            }, this));
        }

        var genericErrors = validationResult.genericErrors;
        if (!_.isEmpty(genericErrors) ) {
            var $message = AJS.messages.error(this.$form, {
                body: usermanagement.form.errorBody({errorArray: genericErrors}),
                insert: 'prepend',
                closeable: false
            });

            this._errors.push({
                message: $message,
                isClientSide: validationResult.isClientSide
            });
        }
    };

    /**
     * Sets this form's submitting state.
     *
     * @param submitting
     * @private
     */
    UMForm.prototype._setSubmitting = function(submitting) {
        this.$submitButton.prop("disabled", submitting);
        if (submitting) {
            this.$throbber.show();
        } else {
            this.$throbber.hide();
        }
    };

    /**
     * Applies defaults and validates the UMForm options.
     *
     * @param {object} options
     * @returns {object}
     * @private
     */
    UMForm.prototype._validateOptions = function(options) {
        var optionsWithDefaults = _.defaults({}, options, {
            dataFormatter: _.identity,
            insertErrorsAutomagically: true,
            presubmitValidator: $.noop
        });

        _.each(['dataFormatter', 'presubmitValidator'], function (f) {
            if (!_.isFunction(optionsWithDefaults[f])) {
                throw new TypeError('Supplied ' + f + ' option is not a function');
            }
        });

        return optionsWithDefaults;
    };

    /**
     * Installs an event listener that runs client-side validation <code>delay</code> milliseconds
     * after a form field has been changed.
     *
     * @param {number} [delay=1000] wait for this many ms before running validation
     * @private
     */
    UMForm.prototype._initChangedFieldsHandler = function(delay) {
        delay = delay || 1000;

        /** field values at form init time */
        var initFields = this._formData().fields;
        this.$form.on("input change", _.debounce($.proxy(function(e) {
            var fieldName = $(e.target).attr('name');
            if (fieldName) {
                var formData = this._formData();
                var nowFields = formData.fields;

                // check if the field has actually changed
                if (this._changedFields[fieldName] || initFields[fieldName] !== nowFields[fieldName]) {
                    this._changedFields[fieldName] = fieldName;
                    this._validateForm(formData);
                }
            }
        }, this), delay));
    };

    /**
     * A form validation result.
     *
     * @constructor
     */
    function FormValidation() {
        this.fieldErrors = {};
        this.genericErrors = [];
    }

    /**
     * Adds an error for the given field.
     *
     * @param {string} name a field name
     * @param {string} message a field error message
     * @param {object} [options]
     */
    FormValidation.prototype.addFieldError = function(name, message, options) {
        options = _.extend({message: message}, options);
        this.fieldErrors[name] = this.fieldErrors[name] || [];

        this.fieldErrors[name].push(options);
    };

    /**
     * Adds a generic error.
     *
     * @param {string} message a generic error message
     */
    FormValidation.prototype.addGenericError = function(message) {
        this.genericErrors.push(message);
    };

    /**
     * Adds an error for the given field.
     */
    FormValidation.prototype.hasErrors = function() {
        return !_.isEmpty(this.fieldErrors) || !_.isEmpty(this.genericErrors);
    };

    window.UserManagement = AJS.namespace('UserManagement');
    window.UserManagement.UMForm = UMForm;
}(AJS.$));
