(function() {
    /**
     * This determines the order to send the form fields to zxcvbn in order to determine password score.
     * This order should match the order on the server.
     * @dict
     */
    var ScoreWordOrder = {
        "name": 1,
        "display-name": 2,
        "email": 3
    };

    var PasswordMeter = AJS.namespace('UserManagement.password.PasswordMeter', Class.extend(/** @lends UserManagement.password.PasswordMeter# */{
        /**
         * Creates a new PasswordMeter. Password meters are designed to be added to form fields, optionally hooking into
         * form validation.
         *
         * Adding the <code>password-score-words</code> class to sibling form fields ensures that their contents are
         * taken into account when scoring the password. This is useful when those fields contain user-specific data
         * such as the full name, username, and other information that should not be allowed in a password.
         *
         * @constructs
         * @param {string|jQuery|HTMLElement} context a selector, jQuery, or HTMLElement
         * @param {string} inputSelector the query selector
         * @param {object} [options] optional configuration options
         * @param {UserManagement.UMForm} options.form the UMForm to add password validation to
         * @param {number} options.policyScoreDeferred the minimum required score specified by the password policy ([0..4]
         * @param {string} [options.zxcvbnUrl] the URL to load the zxcvbn library from
         * @param {boolean} [options.allowEmpty=false] true to allow empty passwords
         */
        init: function (context, inputSelector, options) {
            if (!context) throw new Error('no context');
            if (!inputSelector) throw new Error('no inputSelector');

            this._context = context;
            this._inputSelector = inputSelector;

            /** i18n'ed password strength */
            this._scoreNames = [
                AJS.I18n.getText('usermanagement.passwordpolicy.names.WEAK'),
                AJS.I18n.getText('usermanagement.passwordpolicy.names.FAIR'),
                AJS.I18n.getText('usermanagement.passwordpolicy.names.GOOD'),
                AJS.I18n.getText('usermanagement.passwordpolicy.names.STRONG'),
                AJS.I18n.getText('usermanagement.passwordpolicy.names.VERY_STRONG')
            ];

            this._createMeter(_.pick(options || {}, 'zxcvbnUrl'));
            this._initFormValidation(options);
        },

        /**
         * @returns {number} the password score as a number from 0 (weakest) to 5 (strongest) or undefined
         */
        getScore: function() {
            return this.passwordStrength.getStrength();
        },

        /**
         * @returns {string} the strength text.
         */
        getScoreText: function() {
            return this.passwordStrength.getStrengthText();
        },

        /**
         * Updates the password score based on the current value of the input field.
         */
        updateScore: function() {
            this.passwordStrength.updateStrength();
            return this.getScore();
        },

        /**
         * Creates the password meter.
         *
         * @param {object} opts
         * @param {string} [opts.zxcvbnUrl] where to load zxcvbn.js from
         * @private
         */
        _createMeter: function(opts) {
            opts = _.defaults({}, opts, {
                // This NEEDS TO BE KEPT IN SYNC with the password policy service. Format is ".../git-sha/zxcvbn.js"
                zxcvbnUrl: '//common-admin-cdn.atlassian.com/zxcvbn/f2a8cda/zxcvbn.js'
            });

            var context = AJS.$(this._context)[0];
            var input = AJS.$(this._inputSelector, context);
            if (input.length > 0) {
                // ensure that errors are shown after the password meter that we are about to add
                input.attr('data-insert-error-after', this._inputSelector + '+.passwordstrength');
            }

            // allow one more character over the max length so we can detect when a large
            // password has been pasted and show the field error. without the message it's
            // impossible for a user to figure it out by counting asterisks...
            input.prop('maxlength', PasswordMeter.MAX_LENGTH + 1);

            this.passwordStrength = new PasswordStrength(this._inputSelector, {
                context: context,
                i18nPrefix: '',
                i18nNoValue: '',
                i18nNoZxcvbn: '',
                i18nLevels: Array.prototype.slice.call(this._scoreNames), // take a copy of the array
                zxcvbnUrl: opts.zxcvbnUrl,
                scoreWords: _.bind(this._getScoreWords, this)
            });
        },

        /**
         * Processes the form validation options.
         *
         * @param {object} [opts] optional validation config
         * @param {UserManagement.UMForm} opts.form the UMForm to add password validation to
         * @param {number} opts.policyScoreDeferred the minimum strength specified by the password policy ([0..4]
         * @private
         */
        _initFormValidation: function(opts) {
            opts = _.defaults(opts || {}, {
                allowEmpty: false
            });

            if  (_.isUndefined(opts.form) || _.isUndefined(opts.policyScoreDeferred)) {
                return;
            }

            // adds the validation once the policy strength deferred has been resolved
            opts.policyScoreDeferred.done(_(this._addFormValidator).chain().partial(opts).bind(this).value());
        },

        /**
         * Adds a form validator to check that the password complies with the policy.
         *
         * @param {object} opts validation config
         * @param {UserManagement.UMForm} opts.form
         * @param {boolean} opts.allowEmpty
         * @param {number} policyStrength
         * @private
         */
        _addFormValidator: function(opts, policyStrength) {
            var input = AJS.$(this._inputSelector, this._context);
            if (input.length == 1) {
                var fieldName = input.attr('name');
                if (fieldName) {
                    opts.form.addValidator(this._createValidator(fieldName, policyStrength, opts.allowEmpty));
                }
            }
        },

        /**
         * Returns a new validator for the given field name and policy strength.
         *
         * @param {string} fieldName
         * @param {number} policyStrength
         * @param {boolean} allowEmpty
         * @return {Function}
         *
         * @private
         */
        _createValidator: function(fieldName, policyStrength, allowEmpty) {
            return _.bind(function (validation, fields) {
                var password = fields[fieldName];
                var length = _.size(password);

                if (!allowEmpty && length == 0) {
                    // display the error below the strength indicator
                    validation.addFieldError(fieldName, AJS.I18n.getText('usermanagement.passwordpolicy.empty'));
                } else if (length > PasswordMeter.MAX_LENGTH) {
                    validation.addFieldError(fieldName, AJS.I18n.getText('usermanagement.passwordpolicy.too.long', PasswordMeter.MAX_LENGTH));
                } else if (length > 0) {
                    var score = this.updateScore();
                    if (_.isNumber(score) && score < policyStrength) {
                        validation.addFieldError(fieldName, AJS.I18n.getText('usermanagement.passwordpolicy.try.harder'));
                    }
                }
            }, this);
        },

        /**
         * Returns an array containing all text that is found in form fields marked with class
         * <code>password-score-words</code>.
         *
         * @return {[string]} an array of words to use in scoring
         * @private
         */
        _getScoreWords: function() {
            return _(AJS.$('.password-score-words input', this._context))
                .chain()
                .sortBy(passwordScoreWordFilter)
                .map(extractWords)
                .flatten()
                .filter(function(word) { return !_.isEmpty(word) })
                .value();

            function extractWords(el) {
                var $el = AJS.$(el);
                var tokens = $el.val().split(/(?=\b)/);

                // odd tokens are separators
                return _.filter(tokens, function(_, i) { return i % 2 == 0 });
            }

            function passwordScoreWordFilter(a){
                return ScoreWordOrder[AJS.$(a).attr('name')] || 0;
            }
        }
    }));

    /**
     * Maximum allowed length for passwords.
     *
     * @type {number}
     */
    PasswordMeter.MAX_LENGTH = 128;

    /**
     * Fetches the minimum password policy score from the server and
     * Creates a new PasswordMeter instance for a form.
     *
     * @param {object} opts
     * @param {string|jQuery|HTMLElement} opts.context
     * @param {string} opts.inputSelector
     * @param {UserManagement.UMForm} [opts.addValidationTo] the UMForm to add validation to
     * @param {boolean} [opts.allowEmpty=false] true to allow empty passwords
     * @param {number} [opts.policyScore] the minimum required password score ([0..4])
     *
     * @return {UserManagement.password.PasswordMeter}
     */
    PasswordMeter.addToForm = function(opts) {
        if (_.isEmpty(opts) || _.isEmpty(opts.context) || _.isEmpty(opts.inputSelector)) {
            throw new Error('context and inputSelector are required');
        }

        var options = {};
        if (!_.isUndefined(opts.addValidationTo)) {
            _.extend(options, {
                form: opts.addValidationTo,
                allowEmpty: opts.allowEmpty,
                policyScoreDeferred: _.isUndefined(opts.policyScore) ? PasswordMeter._fetchPolicyFromServer() : AJS.$.when(opts.policyScore)
            });
        }

        return new UserManagement.password.PasswordMeter(opts.context, opts.inputSelector, options);
    };

    /**
     * Fetches the password policy score from the server.
     *
     * @return {jQuery.Deferred}
     * @private
     */
    PasswordMeter._fetchPolicyFromServer = function() {
        return AJS.$.getJSON(AJS.contextPath() + '/rest/um/1/passwordmeter/config').then(function(policy) {
            return policy.policyScore.ranking;
        });
    };
})();
