(function ($, root) {
    var dataTableIndex = 0;
    var i18n = {
            groups: {
                searchPlaceholder: AJS.I18n.getText('usermanagement.groups.searchPlaceholder')
            },
            users: {
                searchPlaceholder: AJS.I18n.getText('usermanagement.users.searchPlaceholder')
            },
            apps: {
                searchPlaceholder: AJS.I18n.getText('usermanagement.apps.searchPlaceholder')
            }
        },
        baseUrl = AJS.contextPath(),
        searchUrls = {
            groups: baseUrl + '/rest/um/1/group/search',
            users: baseUrl + '/rest/um/1/user/search'
        };

    //** create constructor and put it on root
    /**
     * Maintains the pages data table
     * @param type Initial data type that this table should display.
     * @param options UMDataTable options.
     */
    var UMDataTable = function (type, options) {
        var that = this;

        this.setType(type);
        this.options = $.extend({}, UMDataTable.defaults, options);
        if (this.options.parse !== undefined && !$.isFunction(this.options.parse)) {
            throw new TypeError('Supplied parse option is not a function');
        }
        this.itemsToDisplay = this.options.itemsToDisplay;
        this.itemsToRequest = this.itemsToDisplay + 1;
        this.noResultsMessage = this.options.noResultsMessage || AJS.I18n.getText("usermanagement.generic.no.results.found");
        this.inputTextField = this.options.inputTextField || $('#list-search-text');
        this.setQuery(window.location.search);
        this.rowTemplate = this.options.rowTemplate || usermanagement[type].row;
        this.loadedCount = 0;
        this.guid = ++dataTableIndex;
        this.tableTemplateOptions = this.options.tableTemplateOptions;
        this.defaultQueryParams = this.options.defaultQueryParams || {};

        that.init();
    };
    root.UMDataTable = UMDataTable;

    /**
     * Default UMDataTable options
     */
    UMDataTable.defaults = {
        container: '#page-data',
        searchTriggerContainer: undefined,
        itemsToDisplay: 30,
        parse: undefined,
        loadedCallback: $.noop,
        searchFilter: undefined,
        extraParams: {},
        tableTemplateOptions: {}
    };
    UMDataTable.PAGINATION_QUERY_PARAM = 'start-index';
    UMDataTable.SEARCH_QUERY_PARAM = 'filter';
    UMDataTable.MAX_RESULTS_QUERY_PARAM = 'max-results';

    //** instance methods

    /**
     * Initializes necessary properties and handles waiting for the content promise to return.
     */
    UMDataTable.prototype.init = function () {
        var tableDom = usermanagement[this.type].table(this.tableTemplateOptions);
        this.table = $(this.options.container).append(tableDom).find('.data-table');
        this.table.attr("data-table-uuid", window.uuid.v4());
        this.showActions();
        this._addDefaultQueryValues(this.defaultQueryParams);
        this._search();
        this.setupStateChange();
    };

    /**
     * Performs a search by delegating to _search(), either directly or via a statechange event.
     * @param options.pushState Boolean, false if table state should NOT be pushed onto the history stack (e.g. on initial page load, or just refreshing the list)
     */
    UMDataTable.prototype.search = function (options) {
        var data = {
            query: this.getQuery(),
            controllerStateChange: true
        };
        var title = document.title;
        var url = window.location.pathname + this.buildQueryString();

        // History.js does not trigger a statechange event if the new state is the same as the last one.
        // See https://github.com/browserstate/history.js/issues/293
        if (options && options.pushState === false || History.isLastSavedState(History.createStateObject(data, title, url))) {
            this._search();
        } else {
            History.pushState(data, title, url);
        }
    };

    /**
     * Wait for data to load then draw table.
     */
    UMDataTable.prototype._search = function () {
        var that = this;
        var searchUrl = this.getSearchUrl();

        if (!searchUrl) {
            return;
        }

        this.setQueryParam(UMDataTable.MAX_RESULTS_QUERY_PARAM, this.itemsToRequest);
        this.loading(true);

        if (this.currentSearchRequest) {
            this.currentSearchRequest.abort(); //If there is still an outstanding request, kill it and make a new one
        }

        this.currentSearchRequest = $.getJSON(searchUrl + this.buildQueryString()).done(function (data) {
            that.tableContent = that.options.parse ? that.options.parse(data) : data;
            that.drawTable();
        }).always(function (data, status, jqXHR) {
            if (jqXHR === that.currentSearchRequest) {
                that.currentSearchRequest = null;
            }

            if (status !== "abort") {
                that.loading(false);
                that.options.loadedCallback(that);
            }
        });

        // reset the filter to its value without "max-results"
        // we don't actually have to do this since query params are reset on statechange, but just in case...
        this.removeQueryParam(UMDataTable.MAX_RESULTS_QUERY_PARAM);
    };

    /**
     * Drawing the table.
     */
    UMDataTable.prototype.drawTable = function () {
        var rows = [],
            table = this.table.closest("table"),
            filteredTableContent = this.options.searchFilter ? this.options.searchFilter(this.tableContent) : this.tableContent;

        table.find("tbody").attr("loadedcount", this.loadedCount);

        for (var i = 0; i < filteredTableContent.length && i < this.options.itemsToDisplay; i++) {
            rows.push($(this.rowTemplate(filteredTableContent[i]))[0]);
        }
        this.table.empty();

        $(".no-results-message").remove();

        // Get rid of the faux-columns added for "new" lozenges.
        this.table.siblings("thead").find("th.empty-column").remove();

        table.show();

        this.drawPagination();
        if (!rows.length) {
            table.hide();
            table.after($("<p />", { "class": "no-results-message", text: this.noResultsMessage }));
            return;
        }
        this.table.append(rows);

        var $lozenges = this.table.find(".aui-lozenge");
        $lozenges.tooltip && $lozenges.tooltip({aria: true});
    };

    /**
     * Displays loading spinner.
     * @param loading Boolean, true or undefined if page is loading, false if done loading.
     */
    //pull into `app` as opposed to individual controller?
    UMDataTable.prototype.loading = function (loading) {
        loading = typeof loading === "undefined" ? true : loading;
        //if we're loading, we have to determine which spinner kind of spinner to display.
        this.table.closest("table").find("tbody").toggleClass("loading", loading);
        if (loading) {
            // if the spinner is not showing, show it
            if (!this.spinner) {
                this.spinner = $(usermanagement.loading());
                this.spinner.appendTo(
                    typeof this.getQuery()[UMDataTable.PAGINATION_QUERY_PARAM] === "undefined"
                        ? this.options.searchTriggerContainer
                        : $('.aui-nav-pagination', this.options.container)
                );
            }

            // disable pagination buttons
            $(".aui-nav-pagination", this.options.container).find("a").attr("aria-disabled","true");
        } else if (!loading) {
            this.loadedCount++;
            $('#dataActions').find('.aui-icon-wait').remove();
            //We have finished loading, so if there is a spinner, remove it
            if (this.spinner) {
                this.spinner.remove();
                delete this.spinner;
            }

            //re-enable the pagination buttons
            $(".aui-nav-pagination", this.options.container).find("a").removeAttr("aria-disabled");
        }
    };

    (function () {
        // Swallow some activations, to prevent it flickering
        // like a crazy flickering thing.
        var timeout = null;

        UMDataTable.prototype.showBlanket = function () {
            clearTimeout(timeout);

            // Show a semi-transparent blanket over the rows of
            // results which are now "stale".
            var $table = this.getTableElement().closest("table");

            // If it's already there, don't show it again, please!
            if ($table.parent(".table-container").length) {
                return;
            }

            // Guests are coming, put a blanket over that table.
            var $container = $table.wrap("<div />").parent().addClass("table-container");

            // Here comes Prince Michael Jackson II.
            $container.append($("<div />", { "class": "aui-blanket", "aria-hidden": false })).append($table);
        };

        UMDataTable.prototype.hideBlanket = function () {
            // Time to de-blankify.
            var $table = this.getTableElement().closest("table");
            var $tableContainer = $table.parent(".table-container");
            clearTimeout(timeout);

            timeout = setTimeout(function () {
                if ($tableContainer.length) {
                    $tableContainer.find(".aui-blanket").remove();
                    $table.unwrap();
                }
            }, 100);
        };

        UMDataTable.prototype.toggleBlanket = function (show) {
            if (show) {
                this.showBlanket();
            } else {
                this.hideBlanket();
            }
        };
    })();

    /**
     * Displays the data table's actions.
     * @param show Boolean, show true or undefined if the data table actions should be displayed, False to hide them.
     */
    UMDataTable.prototype.showActions = function (show) {
        var type = this.type;
        show = typeof show === "undefined" ? true : show;

        if (show) {
            this.inputTextField.attr('placeholder', i18n[type].searchPlaceholder);
            $('#dataActions').removeClass('hidden');
        } else {
            $('#dataActions').addClass('hidden');
        }
    };

    UMDataTable.prototype.getStartIndex = function () {
        return Math.floor(this.getQuery()[UMDataTable.PAGINATION_QUERY_PARAM]/this.options.itemsToDisplay)*this.options.itemsToDisplay;
    };

    /**
     * Draws pagination below the data table.
     */
    UMDataTable.prototype.drawPagination = function () {
        var startIndex = this.getStartIndex() || 0,
            pageNumber = (startIndex / this.options.itemsToDisplay) + 1,
            //If we got itemsToRequest items back, we there is a `next` page.
            showNext = this.tableContent.length === this.itemsToRequest,
            pagination;

        $('.aui-nav-pagination', this.options.container).remove();
        if (pageNumber === 1 && !showNext) {
            return
        }
        pagination = $(usermanagement.pagination({pageNumber: pageNumber, showNext: showNext, type: this.type}));
        this.table.closest(this.options.container).append(pagination);
    };

    /**
     * Called when a pagination link is clicked.
     * @param e jQuery Event object
     */
    UMDataTable.prototype.paginationEvent = function (e) {
        var clicked = $(e.target).closest('li'),
            currentStartIndex = +this.getStartIndex() || 0,
            next, newStartIndex;
        e.preventDefault();

        //ensure one of the list items was actually clicked and the list item contains an anchor.
        if (!clicked || !clicked.find('a').length) {
            return;
        }
        //Check that the button pressed hasn't been disabled due to loading
        if (clicked.find('a[aria-disabled="true"]').length) {
            return;
        }

        if (clicked.is('.aui-nav-next') || clicked.is('.aui-nav-previous')) {
            next = clicked.is('.aui-nav-next');
            newStartIndex = next ? currentStartIndex + this.options.itemsToDisplay : currentStartIndex - this.options.itemsToDisplay;
        } else {
            clicked = clicked.find('a');
            newStartIndex = (+clicked.attr('href') - 1) * this.options.itemsToDisplay;
        }

        this.setQueryParam(UMDataTable.PAGINATION_QUERY_PARAM, newStartIndex);
        this.search();
    };

    UMDataTable.prototype.highlightNewRows = function (testFn) {
        var lozengifyAllRows = false;
        var $emptyTh = null;
        var $lozenge = null;

        this.table.children("tr").each(function () {
            var $row = $(this);
            var $td = null;
            var lozengifyThisRow = testFn.call(this);

            if (lozengifyThisRow || lozengifyAllRows) {
                $td = $("<td />");
                $row.find("td:first-child").after($td);
            }

            if (lozengifyThisRow) {
                $lozenge = $(aui.lozenges.lozenge({
                    text: AJS.I18n.getText("usermanagement.generic.added"),
                    type: "success",
                    isSubtle: true
                }));
                $td.append($lozenge);
                lozengifyAllRows = true;
                $row.addClass('newly-added');
            }
        });

        // Don't forget the header now! It needs dat extra cell!
        if (lozengifyAllRows) {
            $emptyTh = $("<th />", { "class": "empty-column" });

            // Making the lozenge column a "narrow-column" does sweet FA.
            // Whack the lozenge width on that column. IE demands it is > not >=.
            $emptyTh.width($lozenge.outerWidth() + 1);

            // Yeah, the "table" ain't the table?
            // What can ya do?
            this.table.siblings("thead").find("th:first-child").after($emptyTh);
        }
    };

    /**
     *
     * @param type type of
     */
    UMDataTable.prototype.setType = function (type) {
        this.type = type;
    };
    /**
     * Gives the appropriate search URL based on the table's current type.
     * @returns {*}
     */
    UMDataTable.prototype.getSearchUrl = function () {
        if (this.options.url) {
            if ($.isFunction(this.options.url)) {
                return this.options.url();
            } else {
                return this.options.url;
            }
        } else {
            return searchUrls[this.type];
        }
    };

    UMDataTable.prototype.getExtraParams = function () {
        return $.isFunction(this.options.extraParams) ? this.options.extraParams() : this.options.extraParams;
    };

    /**
     * Parses search into query params.
     * @param search
     */
    UMDataTable.prototype.setQuery = function (search) {
        var splitSearch = search.slice(1).split('&'),
            query = {};

        if (search !== "") {
            $.each(splitSearch, function (i, paramString) {
                var paramPair = paramString.split('=');
                var existing = query[decodeURIComponent(paramPair[0])];
                if (existing) {
                    query[decodeURIComponent(paramPair[0])] = [].concat(existing, decodeURIComponent(paramPair[1]));
                } else {
                    query[decodeURIComponent(paramPair[0])] = decodeURIComponent(paramPair[1]);
                }
            });
        }
        this.query = query;
    };

    /**
     * Get search object
     * @returns {Function}
     */
    UMDataTable.prototype.getQuery = function () {
        return _.extend({}, this.query);
    };

    /**
     * Adds or modify a parameter on the query object.
     * @param key
     * @param value
     */
    UMDataTable.prototype.setQueryParam = function (key, value) {
        if (!value || value.length === 0) {
            delete this.query[key];
        } else {
            this.query[key] = value;
        }
    };
    /**
     * Removes a parameter on the query object.
     * @param key
     */
    UMDataTable.prototype.removeQueryParam = function (key) {
        delete this.query[key];
    };

    /**
     * Sets query parameters.
     * @param query
     */
    UMDataTable.prototype.setQueryParams = function (query) {
        this.query = query || {};
    };

    /**
     * Builds a valid query string from the query object.
     * @returns {string}
     */
    UMDataTable.prototype.buildQueryString = function () {
        var queryString = "",
            first = true,
            itemsToDisplay = this.itemsToDisplay;
        $.each($.extend({}, this.getQuery(), this.getExtraParams()), function (key, value) {
            if (key === UMDataTable.PAGINATION_QUERY_PARAM) {
                value = Math.floor(value / itemsToDisplay) * itemsToDisplay; //To avoid uneven query param problems
            }

            // Handle the "a=a&a=b&a=c" case.
            // This code is some clever shit that turns
            // a non-array value into an array with a
            // single member, to uncomplicate the next
            // part of code. You're welcome.
            value = [].concat(value);

            $.each(value, function (iDontCareAboutThisKey, value) {
                first ? first = false : (queryString += '&');
                queryString += (encodeURIComponent(key) + "=" + encodeURIComponent(value));
            });

        });
        return queryString.length ? '?' + queryString : queryString;
    };

    UMDataTable.prototype._addDefaultQueryValues = function (defaultValues) {
        //If undefined, add the default values
        this.query = _.extend({}, defaultValues, this.query);
    };

    // TODO: refactor as it's not actually the table element.
    // It needs to return this.table.closest("table").
    UMDataTable.prototype.getTableElement = function () {
        return this.table;
    };

    UMDataTable.prototype.setupStateChange = function(){
        var self = this;
        var stateChangeHandler = function(){
            var state = History.getState();
            var query = state.data.query;
            var guid = dataTableIndex;
            if (guid === self.guid) {
                if (query) {
                    self.setQueryParams(query);
                    self.inputTextField.val(query.filter);
                } else {
                    self.removeQueryParam(UMDataTable.PAGINATION_QUERY_PARAM);
                    self.removeQueryParam(UMDataTable.SEARCH_QUERY_PARAM);
                    self.inputTextField.val("");
                }
                self._search();
            } else {
                //The user has navigated away from the page with this data table
                $(window).off('statechange', stateChangeHandler);
            }
        };

        $(window).on('statechange', stateChangeHandler);
    }
})(AJS.$, window.UserManagement = window.UserManagement || {});