(function ($) {

    function CheckboxMultiSelect(options) {
        this.options = _.defaults(options, this.defaults);
        this.$select = $(options.el)
            .change(_.bind(this.renderButton, this))
            .change(_.bind(this._updateClearAll, this));

        this._createFurniture();
        this.renderButton();
        this.renderDropdown();
    }

    CheckboxMultiSelect.prototype = {

        defaults: {
            implicitAllMessage: "All"
        },

        templates: {
            button: aui.checkboxmultiselect.button,
            dropdown: aui.checkboxmultiselect.dropdown
        },

        /**
         * Generates ids unique to this control. This is mainly need as under the covers we use dropdown2, which
         * requires the corresponding dropdown to have a unique id.
         * @private
         * @return {String}
         */
        _generateId: function (suffix) {
            var uniq = this.options.id || this.$select.attr("name") || _.uniqueId();
            return 'aui-checkbox-multiselect-' + uniq + (suffix ? "-" + suffix : "");
        },

        /**
         * Wraps the <select> with out container element
         * @returns {*}
         * @private
         */
        _createContainer: function () {
            var $container = $('<div />')
                .addClass('aui-checkbox-multiselect');

            $container.attr('id', this._generateId());

            if (this.options.styleClass) {
                $container.addClass(this.options.styleClass);
            }

            // Make <select /> child of container
            $container.insertBefore(this.$select);
            this.$select.appendTo($container);
            return $container;
        },

        /**
         * Creates a container element that the dropdown component gets rendered to. Binds to ui events in the dropdown
         * so we can update the <select>
         * @returns {jQuery}
         * @private
         */
        _createDropdown: function () {
            return  $('<div />')
                .addClass('aui-checkbox-multiselect-dropdown aui-dropdown2 aui-style-default')
                .attr('id', this._generateId('dropdown'))
                .on('click', 'a', _.bind(this._handleDropdownSelection, this))
                .on('click', '.js-clear-selections', _.bind(this.clearSelections, this))
                .appendTo(this.$container)
        },


        /**
         * Creates a container element that the button component gets rendered to
         * @returns {jQuery}
         * @private
         */
        _createBtnContainer: function () {
            return  $('<div />')
                .addClass('aui-checkbox-multiselect-btn')
                .attr('id', this._generateId("btn"))
                .appendTo(this.$container)
                .tooltip({
                    title: _.bind(this._getButtonTitle, this)
                });
        },

        /**
         * Gets a comma seperated string of the selected values. When dropdown is open returns empty string as we don't
         * want the title tag to show in that case.
         * @returns {String}
         * @private
         */
        _getButtonTitle: function () {
            return this.$dropdown.is(":visible") ? "" : this.getSelectedLabels().join(", ");
        },

        /**
         * Creates all the different 'container' elements, the the individual components (button, dropdown) get rendered to.
         * @private
         */
        _createFurniture: function () {
            this.$container = this._createContainer();
            this.$dropdown = this._createDropdown();
            this.$btnContainer = this._createBtnContainer();
        },

        /**
         * Handles when user clicks an item in the dropdown list. Either selects or unselects the corresponding
         * option in the <select>.
         * @private
         */
        _handleDropdownSelection: function (e) {
            var $a = $(e.currentTarget);
            var value = $a.attr('data-value');
            this.toggleSelected(value, !$a.hasClass('aui-dropdown2-checked'));
            e.preventDefault();
        },

        /**
         * Converts a jQuery collection of <option> elements into an object that describes them.
         * @param {jQuery} $options
         * @returns {Array<Object>}
         * @private
         */
        _mapOptionDescriptors: function ($options) {
            return $options.map(function () {
                var $option = $(this);
                return {
                    value: $option.val(),
                    label: $option.text(),
                    icon: $option.data('icon'),
                    styleClass: $option.data('styleClass'),
                    title: $option.attr('title'),
                    disabled: $option.attr('disabled'),
                    selected: $option.attr('selected'),
                    badge: $option.data("badge")
                };
            });
        },

        /**
         * Enables 'clear all' button if there are any selected <option>s, otherwise disables it.
         * @private
         */
        _updateClearAll: function () {
            this.$dropdown.find('.js-clear-selections').prop('disabled', _.bind(function () {
                return this.getSelectedDescriptors().length < 1;
            }, this));
        },

        /**
         * Selects or unselects the <option> corresponding the given value.
         * @param value - value of option to update
         * @param {Boolean} selected - select or unselect it.
         */
        toggleSelected: function (value, selected) {
            var $toUpdate = this.$select.find("option").filter(function () {
                var $this = $(this);
                return $this.attr("value") === value && $this.prop("selected") != selected;
            });
            if ($toUpdate.length) {
                $toUpdate.prop("selected", selected);
                this.$select.trigger("change");
            }
        },

        /**
         * Gets descriptor for selected options, the descriptor is an object that contains meta information about
         * the option, such as value, label, icon etc.
         * @returns Array<Object>
         */
        getDescriptors: function () {
            return this._mapOptionDescriptors(this.$select.find('option'));
        },

        /**
         * Gets descriptor for selected options, the descriptor is an object that contains meta information about
         * the option, such as value, label, icon etc.
         * @returns Array<Object>
         */
        getSelectedDescriptors: function () {
            return this._mapOptionDescriptors(this.$select.find('option:selected'));
        },

        /**
         * Gets the innerText of the selected options
         * @returns Array<String>
         */
        getSelectedLabels: function () {
            return _.map(this.getSelectedDescriptors(), function (descriptor) {
                return descriptor.label;
            });
        },

        /**
         * If nothing is selected, we take this to mean that everything is selected.
         * @returns Boolean
         */
        isImplicitAll: function () {
            return this.getSelectedDescriptors().length === 0;
        },

        /**
         * Renders dropdown with list of items representing the selected or unselect state of the <option>s in <select>
         */
        renderDropdown: function () {
            this.$dropdown.html(this.templates.dropdown({
                items: this.getDescriptors(),
                isImplicitAll: this.isImplicitAll()
            }));
            this._updateClearAll();
        },

        /**
         * Renders button with the selected <option>'s innerText in a comma seperated list. If nothing is selected 'All'
         * is displayed.
         */
        renderButton: function () {
            var selectedLabels = this.getSelectedLabels();
            var label = this.isImplicitAll() ? this.options.implicitAllMessage : selectedLabels.join(', ');
            this.$btnContainer.html(aui.checkboxmultiselect.button({
                text: label,
                dropdownId: this._generateId('dropdown')
            }));
        },
        /**
         * Unchecks all items in the dropdown and in the <select>
         */
        clearSelections: function () {
            this.$select.val([]).trigger('change');
            this.$dropdown.find('.aui-dropdown2-checked,.checked').removeClass('aui-dropdown2-checked checked');
        },
        /**
         * Adds an option to the <select>
         * @param descriptor
         */
        addOption: function (descriptor) {
            $("<option />").attr({
                "value": descriptor.value,
                "data-icon": descriptor.icon,
                "data-badge": descriptor.badge,
                "disabled": descriptor.disabled,
                "selected": descriptor.selected,
                "title": descriptor.title
            })
            .text(descriptor.label)
            .appendTo(this.$select);
            this.renderButton();
            this.renderDropdown();
        },
        /**
         * Removes options matching value from <select>
         * @param value
         */
        removeOption: function (value) {
            this.$select.find("[value='" + value + "']").remove();
            this.renderButton();
            this.renderDropdown();
        }
    };

    AJS.$.fn.checkboxMultiSelect = function (options) {
        options = options || {};
        var checkboxMultiSelect = new CheckboxMultiSelect({
            el: this,
            implicitAllMessage: options.implicitAllMessage
        });
        this.data("checkbox-multi-select", checkboxMultiSelect);
        return this;
    };

})(AJS.$);