(function (root, factory) {
    if (typeof define === 'function' && define.amd) {
        define(function () {
            return (root.PasswordStrength = factory(root));
        });
    } else {
        root.PasswordStrength = factory(root);
    }
}(this, function (window) {
    'use strict';

    var doc = window.document;
    var MIN_I18N_LEVELS = 5;

    /**
     * PasswordStrength: helping you create passwords that stand the test of time. Callers should make sure to create PasswordStrength
     * instances after "DOMContentLoaded" only.
     *
     * @param selector
     * @param options
     * @param {(function():[string])|[string]} [options.scoreWords] optional words to use when scoring passwords (array or function)
     * @constructor
     */
    function PasswordStrength(selector, options) {
        options || (options = {});
        this.i18nLevels = (options.i18nLevels && options.i18nLevels.length >= MIN_I18N_LEVELS) ? options.i18nLevels : ['Terrible', 'Weak', 'Average', 'Strong', 'Awesome!'];
        this.i18nNoValue = getString(options.i18nNoValue, 'No password entered');
        this.i18nNoZxcvbn = getString(options.i18nNoZxcvbn, 'Unavailable');
        this.i18nPrefix = getString(options.i18nPrefix, 'Password strength: ');
        this.zxcvbnUrl = options.zxcvbnUrl || '//s3.amazonaws.com/common-admin-cdn/zxcvbn/f4e0c24/zxcvbn.js';
        /** words to take into account when scoring a password */
        this._scoreWords = options.scoreWords;
        this.context = options.context || doc;
        this.el = this.context.querySelector(selector);

        this._classPrefix = 'passwordstrength';
        this._unavailable = 'unavailable';
        this._none = 'none';
        this._matcher = new RegExp(this._classPrefix + '-([0-4]|' + this._none + '|' + this._unavailable + ')');

        // Add test for ordering
        this.meterEl = this.createMeter();

        this.loadZxcvbn();
        this.init();
    }

    PasswordStrength.prototype.init = function () {
        this.el.parentNode.insertBefore(this.meterEl, this.el.nextSibling);
        addListener(this.el, 'keyup', this.updateStrength, this);
    };

    PasswordStrength.prototype.destroy = function () {
        this.el && this.el.removeEventListener('keyup', this.updateStrength);
    };

    PasswordStrength.prototype.loadZxcvbn = function () {
        var s = doc.createElement('script');
        s.type = 'text/javascript';
        s.async = true;
        s.src = this.zxcvbnUrl;
        var x = doc.getElementsByTagName('script')[0];
        x.parentNode.insertBefore(s, x);
        addListener(s, 'load', this.updateStrength, this);
    };

    PasswordStrength.prototype.createMeter = function () {
        var meter = doc.createElement('div');
        meter.className = this._classPrefix + ' ' + this._classPrefix + '-none';
        meter.innerHTML = this.getStrengthText();
        return meter;
    };

    PasswordStrength.prototype.updateStrength = function () {
        var val = this.el.value;
        var score;
        if (typeof zxcvbn === 'undefined') {
            score = val !== '' ? this._unavailable : this._none;
        } else {
            score = val !== '' ? zxcvbn(val, this._getScoreWords()).score : this._none;
        }

        this._score = typeof score === 'number' ? score : null;
        this.meterEl.innerHTML = this.getStrengthText(score);
        this.meterEl.className = this.meterEl.className.replace(this._matcher, this._classPrefix + '-' + score);
    };

    /**
     * @return {number|*} the password strength, or undefined if it hasn't been scored yet
     */
    PasswordStrength.prototype.getStrength = function () {
        return this._score;
    };

    PasswordStrength.prototype.getStrengthText = function (score) {
        if (score === this._unavailable) {
            return this.i18nPrefix + this.i18nNoZxcvbn;
        } else {
            return (typeof score === 'number' ? this.i18nLevels[score] : '');
        }
    };

    /**
     * @returns {Array} an array of strings to pass to zxcvbn
     * @private
     */
    PasswordStrength.prototype._getScoreWords = function () {
        if (typeof this._scoreWords === 'function') {
            return this._scoreWords();
        }

        if (this._scoreWords instanceof Array) {
            return this._scoreWords;
        }

        return [];
    };

    function addListener(el, event, fn, scope) {
        var bound = function () { fn.apply(scope, Array.prototype.slice.call(arguments)); };
        el.addEventListener ? el.addEventListener(event, bound) : el.attachEvent('on' + event, bound);
    }

    function getString(string, defaultValue) {
        if (typeof string !== 'undefined' && string != null) {
            return string;
        }

        return defaultValue;
    }

    return PasswordStrength;
}));

