(function($, root) {

    root.helpers.users = {

        /**
         * Formats individual attributes ready for display in the ui.
         * @param {Array} attributes.attributes - The actual attributes
         * @return {Object} - the formatted attributes
         */
        formatUserAttributes: function (attributes) {
            attributes = _.object(_.map(attributes && attributes.attributes, function (attr) {
                var value = attr.values[0];
                return [attr.name, $.isNumeric(value) ? +value : value];
            }));

            attributes.lastAuthenticated = attributes.lastAuthenticated && root.helpers.formatDateTime(attributes.lastAuthenticated);
            attributes.lastActive = attributes.lastActive && root.helpers.formatDateTime(attributes.lastActive);
            if (attributes.invitationSendDate) {
                attributes.invitationSendDate = root.helpers.relativeTime(attributes.invitationSendDate, function (time) {
                    return root.helpers.isFuture(attributes.invitationExpiryDate) ? AJS.I18n.getText("usermanagement.users.invitation.sent", time) : AJS.I18n.getText("usermanagement.users.invitation.expired", time);
                });
            }

            return attributes;
        },

        /**
         * If there are more than 2 options, creates a user filter picker.
         * @param {jQuery} $target - where to append to
         */
        setupUserFilter: function ($target) {
            // Add user filter only if there are service desk customers in DB.
            $.getJSON(AJS.contextPath() + "/rest/um/1/user/hasUsersOfType", {
                userFilter: "synch.servicedesk.requestor"
            }).done(function (hasSdUser) {
                if (hasSdUser) {
                    $(aui.form.select({
                        id: "user-types",
                        isMultiple: true,
                        options: [
                            { text: AJS.I18n.getText("usermanagement.users.users"), value: "users" },
                            { text: AJS.I18n.getText("usermanagement.users.sd.requesters"), value: "synch.servicedesk.requestor" }
                        ]
                    }))
                    .appendTo($target)
                    .checkboxMultiSelect({
                        implicitAllMessage: AJS.I18n.getText("usermanagement.users.sd.both")
                    });
                }
            });
        },

        setupActiveFilter: function ($target) {
                if (adminApp.isFeatureEnabled("um.disable.user")) {
                    $(aui.form.select({
                        id: "user-active-filter",
                        isMultiple: true,
                        options: [
                            { text: AJS.I18n.getText("usermanagement.users.filter.active"), value: "true" },
                            { text: AJS.I18n.getText("usermanagement.users.filter.inactive"), value: "false" }
                        ]
                    }))
                    .appendTo($target)
                    .checkboxMultiSelect({
                        implicitAllMessage: AJS.I18n.getText("usermanagement.users.active.both")
                    });

                    $('a[data-value="true"]').click(); //Set active only to be default
                }
        },

        /**
         * Gets meta data to render user lozenges. User lozenges are badges that indicate special attributes of a user,
         * for example if they are a service desk requester.
         * @param {Object} user
         * @returns {Array}
         */
        getLozenges: function (user) {
            var lozenges = [];
            if (user.lozenge) {
                lozenges.push(user.lozenge);
            }
            return lozenges;
        },

        /**
         * The data we get back from crowd isn't exactly formatted the way we want. So this function transforms it.
         * - Pulls useful parameters that are buried deep into the object into a more accessible location.
         * - Formats dates into a human readable format
         * - Replaces - with _ in object names so they can easily accessible via JavaScript
         * - Creates badges based on special user attributes for example if they are a service desk requester.
         *
         * @param {Object} user - json translation of com.atlassian.crowd.plugin.rest.entity.UserEntity
         * @return {Object} user json ready for display in UI
         */
        sanitiseAndMapUserData: function(user) {
            var cleanUser = root.util.underscoreParamNames(user);
            cleanUser.lozenge = user.lozenge || null;
            cleanUser.attributes = root.helpers.users.formatUserAttributes(user.attributes);
            cleanUser.createdDate = root.helpers.formatDateTime(user.createdDate);
            cleanUser.aidAccount = user.aidAccount;
            return cleanUser;
        },

        setupMultipleUserPicker: function(options) {

            options = $.extend({}, options);

            var $element = $(options.element);
            var PER_PAGE = 10;

            // Can be removed and placed in Soy template
            // once the right param works in AUI.
            $element.addClass("long-field");

            $element.auiSelect2({
                placeholder: AJS.I18n.getText("usermanagement.user.picker.placeholder"),
                multiple: true,
                openOnEnter: false,
                ajax: {
                    url: AJS.contextPath() + "/rest/um/1/user/search",
                    dataType: 'json',
                    quietMillis: 100,
                    data: root.helpers.select2.groupsUsersDataTranslator(PER_PAGE, options.userFilters),
                    results: function (users) {
                        users = $.map(users, function(user) {
                            return {
                                text: AJS.I18n.getText("usermanagement.user.picker.item", user.name, user["display-name"]),
                                id: user.name
                            };
                        });
                        return {results: users, more: users.length == PER_PAGE};
                    }
                },
                initSelection: function(element, callback){
                    // ignore the username, just return the users
                    callback(options.defaultSelection);
                },
                // I18n
                formatNoMatches: function() {
                    return AJS.I18n.getText("usermanagement.user.picker.no.users.found");
                },
                formatSearching: function() {
                    return AJS.I18n.getText("usermanagement.user.picker.searching");
                },
                formatLoadMore: function() {
                    return AJS.I18n.getText("usermanagement.user.picker.loading.more");
                }
            });

            root.helpers.select2.makePlaceholderBehaviourLikeNative($element);

            // transform users to username
            if (options.defaultSelection) {
                $element.auiSelect2('val', $.map(options.defaultSelection, function(value) {
                    return value.id;
                }));
            }

            if (options.autoFocus) {
                // setTimeout required for IE8 otherwise the element may not always gain focus
                setTimeout(function () {
                    root.helpers.select2.getTransformedElementReference($element).focus();
                }, 0);
            }

            return {
                getUsers: function() {
                    return $element.auiSelect2("data");
                },
                clearSelection: function() {
                    return $element.auiSelect2("val", "");
                }
            };

        },

        setupMultipleEmailPicker: function($el, options) {

            options = $.extend({}, options, {
                placeholder: AJS.I18n.getText("usermanagement.users.multiple.email.address.placeholder"),
                multiple: true,
                tags: [],
                tokenSeparators: [",", " ", ";"],
                // Workaround: ensure we never display the "matched" results,
                // as none will ever exist.
                dropdownCss: { display: "none" },
                openOnEnter: false,
                selectOnBlur: true
            });

            // Can be removed and placed in Soy template
            // once the right param works in AUI.
            $el.addClass("long-field");

            $el.auiSelect2(options);

            var $transformedElement = root.helpers.select2.getTransformedElementReference($el);

            $transformedElement.on("input", function() {
                // Clean up any existing errors.
                $el.nextAll(".error").remove();
            });

            root.helpers.select2.makePlaceholderBehaviourLikeNative($el);

            // IE8 will explode if you try to focus on a disabled element - http://bugs.jquery.com/ticket/10859
            if (!$transformedElement.prop("disabled")) {
                $transformedElement.focus();
            }

            return {
                getEmails: function() {
                    return $el.auiSelect2("val");
                },
                on: $.proxy($.fn.on, $el)
            };

        }
    };

})(AJS.$, window.UserManagement = window.UserManagement || {});