package com.atlassian.crowd.directory.synchronisation.cache;

import com.atlassian.crowd.directory.DirectoryCacheChangeOperations;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.InternalDirectoryGroup;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;

public abstract class AbstractGroupActionStrategy implements GroupActionStrategy {

    private Logger logger = LoggerFactory.getLogger(AbstractGroupActionStrategy.class);

    protected static final DirectoryCacheChangeOperations.GroupsToAddUpdateReplace NO_OP =
            new DirectoryCacheChangeOperations.GroupsToAddUpdateReplace(ImmutableSet.of(), ImmutableSet.of(), ImmutableMap.of());

    protected DirectoryCacheChangeOperations.GroupsToAddUpdateReplace replaceGroup(final InternalDirectoryGroup internalGroup, final Group remoteGroup) {
        return new DirectoryCacheChangeOperations.GroupsToAddUpdateReplace(ImmutableSet.of(),
                ImmutableSet.of(),
                ImmutableMap.of(internalGroup.getName(), makeGroupTemplate(remoteGroup)));
    }

    protected DirectoryCacheChangeOperations.GroupsToAddUpdateReplace updateGroup(final InternalDirectoryGroup internalGroup, final Group remoteGroup) {
        final GroupTemplate groupToUpdate = makeGroupTemplate(remoteGroup);
        groupToUpdate.setName(internalGroup.getName());
        return new DirectoryCacheChangeOperations.GroupsToAddUpdateReplace(ImmutableSet.of(),
                ImmutableSet.of(groupToUpdate),
                ImmutableMap.of());
    }

    protected DirectoryCacheChangeOperations.GroupsToAddUpdateReplace addGroup(final Group remoteGroup) {
        return new DirectoryCacheChangeOperations.GroupsToAddUpdateReplace(ImmutableSet.of(makeGroupTemplate(remoteGroup)), ImmutableSet.of(), ImmutableMap.of());
    }

    private static GroupTemplate makeGroupTemplate(final Group group) {
        GroupTemplate template = new GroupTemplate(group);
        template.setDescription(group.getDescription());
        return template;
    }

    protected boolean wasGroupUpdatedAfterSearchStart(Group remoteGroup, InternalDirectoryGroup internalGroup, Date syncStartDate, long directoryId) {
        if (internalGroup.getUpdatedDate() == null) {
            // This can happen if the Crowd Embedded SPI is not implemented correctly.
            logger.warn("group [ {} ] in directory [ {} ] has no updated date", remoteGroup.getName(), directoryId);
            // ALWAYS do this comparison with the real millis as these may be SQL Timestamps which will throw away millis whenever they feel like it.
        } else if (syncStartDate != null && internalGroup.getUpdatedDate().getTime() > syncStartDate.getTime()) {
            // Don't update this group, it was changed locally after we started our search.
            // Any anomalies will catch up on the next synchronization.
            // This will not always be the case, for example in Azure AD if the next incremental synchronisation will not
            // return the group, we will miss the changes
            logger.debug("group [ {} ] in directory [ {} ] modified after synchronisation start, skipping", remoteGroup.getName(), directoryId);
            return true;
        }
        return false;
    }
}
