package com.atlassian.crowd.search.util;

import com.atlassian.crowd.embedded.api.Query;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;

import java.util.List;
import java.util.Optional;

/**
 * Utility class for executing split queries and merging results.
 */
public class QuerySplitter {
    /**
     * Splits the query if needed, executes it, aggregates and returns results. Query is split if the number
     * of OR conditions is higher than {@code maxRestrictionsPerQuery}. Only top level condition split is supported.
     * @param query original query to be executed
     * @param searcher query results provider
     * @param maxRestrictionsPerQuery maximum number of restrictions allowed in the single query
     * @param <T> result type
     * @return results of the original {@code query}
     * @throws OperationFailedException rethrows exception from {@code searcher}
     */
    public static <T, E extends Exception> List<T> batchConditionsIfNeeded(
            EntityQuery<T> query, EntitySearcher<T, E> searcher, int maxRestrictionsPerQuery) throws E {
        Optional<List<EntityQuery<T>>> split = query.splitOrRestrictionIfNeeded(maxRestrictionsPerQuery);
        return split.isPresent() ? runInBatches(query, split.get(), searcher) : searcher.search(query);
    }

    /**
     * Splits the query if needed, executes it, aggregates and returns results. Query is split if the size
     * of {@link MembershipQuery#getEntityNamesToMatch()} is higher than {@code maxBatchSize}.
     * @param query original query to be executed
     * @param searcher query results provider
     * @param maxBatchSize maximum number of {@link MembershipQuery#getEntityNamesToMatch()} allowed in the single query
     * @param <T> result type
     * @return results of the original {@code query}
     * @throws OperationFailedException rethrows exception from {@code searcher}
     */
    public static <T, E extends Exception> List<T> batchNamesToMatchIfNeeded(
            MembershipQuery<T> query, MembershipSearcher<T, E> searcher, int maxBatchSize) throws E {
        return query.getEntityNamesToMatch().size() > maxBatchSize
                ? runInBatches(query, query.splitEntityNamesToMatch(maxBatchSize), searcher)
                : searcher.search(query);
    }

    public interface EntitySearcher<T, E extends Exception> extends Searcher<T, EntityQuery<T>, E> {
    }

    public interface MembershipSearcher<T, E extends Exception> extends Searcher<T, MembershipQuery<T>, E> {
    }

    public interface Searcher<T, Q extends Query<T>, E extends Exception> {
        List<T> search(Q query) throws E;
    }

    private static <T, Q extends Query<T>, E extends Exception> List<T> runInBatches(
            Q original, List<Q> split, Searcher<T, Q, E> searcher) throws E {
        ResultsAggregator<T> aggregator = ResultsAggregators.with(original);
        for (Q singleQuery : split) {
            aggregator.addAll(searcher.search(singleQuery));
        }
        return aggregator.constrainResults();
    }
}
