package com.atlassian.crowd.manager.mail;

import com.google.common.collect.Iterables;

import javax.activation.DataSource;
import javax.annotation.Nullable;
import javax.mail.internet.InternetAddress;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;

/**
 * Represents an email message
 *
 * @since v3.3.0
 */
public interface EmailMessage {

    /**
     * @return Address of sender of this email, if overridden; by default sender is configured globally for the instance
     */
    Optional<InternetAddress> getFrom();

    /**
     * @return Address of recipient of the email message
     * @deprecated since 3.7 use {@link #getTo()} as it's possible to have multiple recipients of a single e-mail
     */
    @Nullable
    default InternetAddress getRecipientAddress() {
        return Iterables.getFirst(getTo(), null);
    }

    /**
     * @return Recipients of this e-mail of type "To" (primary recipients)
     * @since 3.7
     */
    Collection<InternetAddress> getTo();

    /**
     * @return Recipients of this e-mail of type "Cc" (carbon copy)
     * @since 3.7
     */
    Collection<InternetAddress> getCc();

    /**
     * @return Recipients of this e-mail of type "Bcc" (blind carbon copy)
     * @since 3.7
     */
    Collection<InternetAddress> getBcc();

    /**
     * @return Addresses sent as "Reply-to".
     * @since 3.7
     */
    Collection<InternetAddress> getReplyTo();

    /**
     * @return Body of the email message
     */
    String getBody();

    /**
     * @return Subject of the email message
     */
    String getSubject();

    /**
     * @return Headers of the email message
     */
    Map<String, String> getHeaders();

    /**
     * @return Attachments of the email message
     */
    Map<String, DataSource> getAttachments();
}
