package com.atlassian.crowd.manager.permission;


import com.atlassian.crowd.exception.ApplicationNotFoundException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.permission.UserPermission;
import com.atlassian.crowd.model.page.Page;

import java.util.List;

import javax.annotation.Nonnull;

public interface UserPermissionAdminService {
    /**
     * Set the granted permissions for each directoryGroup to permission. All other permissions will be revoked.
     *
     * Groups must be in directories that are mapped to the "crowd" application
     *
     * @param directoryGroupPairs list of groups to set permissions on
     * @param permission          permission to set to
     * @throws UserPermissionDowngradeException if the user can not remove a permission because
     *                                          doing so would downgrade their own permissions
     */
    void setPermissionForGroups(List<? extends DirectoryGroup> directoryGroupPairs, UserPermission permission) throws DirectoryNotFoundException, OperationFailedException, ApplicationNotFoundException, UserPermissionDowngradeException, AnonymousUserPermissionException;

    /**
     * Remove all permissions from the given group.
     *
     * @param group group to remove permissions from
     * @throws UserPermissionDowngradeException if the user can not remove a permission because
     *                                          doing so would downgrade their own permissions
     */
    void revokePermissionsForGroup(DirectoryGroup group) throws DirectoryNotFoundException, OperationFailedException, ApplicationNotFoundException, UserPermissionDowngradeException, AnonymousUserPermissionException;

    /**
     * Find all groups with an explicit permission
     *
     * Note: there are no authentication checks in this method
     *
     * @param prefix filter group names by name prefix
     * @param start  index to start page at
     * @param limit  max number of results to return or 0 for no limit
     * @return the highest permission for each group with an explicit permission. If
     * a group has ADMIN and SYS_ADMIN, only return SYS_ADMIN
     */
    Page<PermittedGroup> findGroupsWithPermissionByPrefix(@Nonnull String prefix, int start, int limit) throws UserPermissionException, AnonymousUserPermissionException;

    /**
     * Find all groups with an explicit permission
     *
     * Note: there are no authentication checks in this method
     *
     * @param start index to start page at
     * @param limit max number of results to return or 0 for no limit
     * @return the highest permission for each group with an explicit permission. If
     * a group has ADMIN and SYS_ADMIN, only return SYS_ADMIN
     */
    Page<PermittedGroup> findGroupsWithPermission(int start, int limit) throws UserPermissionException, AnonymousUserPermissionException;

    /**
     * List all groups that the Crowd console application can see, filtered by prefix.
     *
     * Groups will be sorted by (directory name, group name) in dictionary order.
     *
     * @param start index to start page at
     * @param limit max number of results to return or 0 for no limit
     */
    Page<DirectoryGroup> findGroupsByPrefix(@Nonnull String prefix, int start, int limit) throws AnonymousUserPermissionException;

    /**
     * List all groups that the Crowd console application can see.
     *
     * Groups will be sorted by (directory name, group name) in dictionary order.
     *
     * @param start index to start page at
     * @param limit max number of results to return or 0 for no limit
     */
    Page<DirectoryGroup> findGroups(int start, int limit) throws AnonymousUserPermissionException;
}
