package com.atlassian.crowd.util.cache;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.Cache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Helper class to create expiring maps.
 */
public class LocalCacheUtils {
    private static final Logger log = LoggerFactory.getLogger(LocalCacheUtils.class);

    private LocalCacheUtils() {
    }

    public static <K, V> ConcurrentMap<K, V> createExpiringAfterAccessMap(Duration ttl) {
        return LocalCacheUtils.<K, V>createExpiringAfterAccessCache(ttl).asMap();
    }

    public static <K, V> ConcurrentMap<K, V> createExpiringAfterAccessMap(Duration ttl, ScheduledExecutorService cleanupPool) {
        Cache<K, V> cache = createExpiringAfterAccessCache(ttl);

        cleanupPool.scheduleWithFixedDelay(
                () -> {
                    try {
                        cache.cleanUp();
                    } catch (Exception e) {
                        log.info("Error while cleaning up cache", e);
                    }
                },
                ttl.getSeconds(), ttl.getSeconds(), TimeUnit.SECONDS);
        return cache.asMap();
    }

    private static <K, V> Cache<K, V> createExpiringAfterAccessCache(Duration ttl) {
        return CacheBuilder.newBuilder()
                .expireAfterAccess(ttl)
                .build();
    }
}
