package com.atlassian.crowd.search.hibernate.audit;

import com.atlassian.crowd.audit.AuditLogEntityType;
import com.atlassian.crowd.audit.AuditLogEventSource;
import com.atlassian.crowd.audit.AuditLogEventType;
import com.atlassian.crowd.audit.query.AuditLogChangesetProjection;
import com.atlassian.crowd.audit.query.AuditLogQuery;
import com.atlassian.crowd.audit.query.AuditLogQueryAuthorRestriction;
import com.atlassian.crowd.audit.query.AuditLogQueryEntityRestriction;
import com.atlassian.crowd.search.hibernate.HQLQuery;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestriction;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Translates an instance of {@link AuditLogQuery} into a {@link HQLQuery}, that can later be executed.
 */
public class AuditLogQueryTranslator {

    static final String DIRECTORY_ALIAS = "directoryentities";
    static final String USER_ALIAS = "userentities";
    static final String GROUP_ALIAS = "groupentities";
    static final String AUDIT_LOG_ENTITY_NAME = "entities";
    static final String CHANGESET_ALIAS = "chset";
    static final String APPLICATION_ALIAS = "applicationentities";
    static final String CHANGESET_PROJECTION_KEY_ALIAS = "projection";

    static final String CHANGESET_TIMESTAMP_PROPERTY = "timestamp";
    static final String CHANGESET_ENTITY = "AuditLogChangesetEntity";
    static final String CHANGESET_AUTHOR_TYPE_PROPERTY = "authorType";
    static final String ENTITY_ID_PROPERTY = "entityId";
    static final String ENTITY_NAME_PROPERTY = "entityName";
    static final String ENTITY_TYPE_PROPERTY = "entityType";
    static final String ENTITY_PRIMARY_PROPERTY = "primary";
    static final String EVENT_TYPE_PROPERTY = "eventType";
    static final String EVENT_SOURCE_PROPERTY = "source";
    static final String AUTHOR_ID_PROPERTY = "authorId";
    static final String AUTHOR_NAME_PROPERTY = "authorName";

    public HQLQuery asHQL(AuditLogQuery query) {
        final HQLQuery hqlQuery = new HQLQuery();
        appendFrom(hqlQuery, query);
        appendRestrictions(hqlQuery, query);
        configureStartIndexAndMaxResults(query, hqlQuery);
        appendOrderBy(query, hqlQuery);
        return hqlQuery;
    }

    private void configureStartIndexAndMaxResults(AuditLogQuery query, HQLQuery hqlQuery) {
        hqlQuery.offsetResults(query.getStartIndex());
        hqlQuery.limitResults(query.getMaxResults());
    }

    private void appendOrderBy(AuditLogQuery query, HQLQuery hqlQuery) {
        if (query.getProjection() == null) {
            hqlQuery.appendOrderBy(CHANGESET_ALIAS + "." + CHANGESET_TIMESTAMP_PROPERTY + " DESC, " + CHANGESET_ALIAS + ".id DESC");
        } else {
            hqlQuery.appendOrderBy(CHANGESET_PROJECTION_KEY_ALIAS + " ASC");
        }
    }

    private void appendRestrictions(HQLQuery hqlQuery, AuditLogQuery query) {
        final BooleanHqlRestriction allClauses = createTopLevelClause(query);
        allClauses.visit(hqlQuery);
    }

    private BooleanHqlRestriction createTopLevelClause(AuditLogQuery query) {
        final ImmutableList.Builder<Restriction> queryRestrictions = new ImmutableList.Builder<>();
        queryRestrictions.add(createTimeRestrictions(query));
        queryRestrictions.add(createActionsRestriction(query.getActions()));
        queryRestrictions.add(createSourcesRestriction(query.getSources()));
        queryRestrictions.add(createAuthorsRestriction(query.getAuthors()));
        queryRestrictions.add(createUsersRestriction(query.getUsers()));
        queryRestrictions.add(createGroupsRestriction(query.getGroups()));
        queryRestrictions.add(createApplicationsRestriction(query.getApplications()));
        queryRestrictions.add(createDirectoriesRestriction(query.getDirectories()));
        return new BooleanHqlRestriction(BooleanRestriction.BooleanLogic.AND, queryRestrictions.build());
    }

    private Restriction createDirectoriesRestriction(List<AuditLogQueryEntityRestriction> directories) {
        return createEntityRestriction(directories, DIRECTORY_ALIAS, AuditLogEntityType.DIRECTORY);
    }

    private Restriction createUsersRestriction(List<AuditLogQueryEntityRestriction> users) {
        return createEntityRestriction(users, USER_ALIAS, AuditLogEntityType.USER);
    }

    private Restriction createGroupsRestriction(List<AuditLogQueryEntityRestriction> groups) {
        return createEntityRestriction(groups, GROUP_ALIAS, AuditLogEntityType.GROUP);
    }

    private Restriction createApplicationsRestriction(List<AuditLogQueryEntityRestriction> groups) {
        return createEntityRestriction(groups, APPLICATION_ALIAS, AuditLogEntityType.APPLICATION);
    }

    private Restriction createEntityRestriction(List<AuditLogQueryEntityRestriction> entities,
                                                String alias, AuditLogEntityType entityType) {
        return createClauseWithJoin(entities, "INNER JOIN " + CHANGESET_ALIAS + "." + AUDIT_LOG_ENTITY_NAME + " AS " + alias, alias, entityType);
    }

    private Restriction createAuthorsRestriction(List<AuditLogQueryAuthorRestriction> authors) {
        return createRestrictionGroup(appendBooleanLogicClauseForCollection(
                authors,
                createAuthorRestrictionMapper(CHANGESET_ALIAS + "." + AUTHOR_ID_PROPERTY, CHANGESET_ALIAS + "." + AUTHOR_NAME_PROPERTY)
        ));
    }

    private Restriction createActionsRestriction(Collection<AuditLogEventType> actions) {
        return createRestrictionGroup(appendBooleanLogicClauseForCollection(
                actions,
                action -> new SimpleRestriction(CHANGESET_ALIAS + "." + EVENT_TYPE_PROPERTY, "=", action)
        ));
    }

    private Restriction createSourcesRestriction(Collection<AuditLogEventSource> sources) {
        return createRestrictionGroup(appendBooleanLogicClauseForCollection(
                sources,
                source -> new SimpleRestriction(CHANGESET_ALIAS + "." + EVENT_SOURCE_PROPERTY, "=", source)
        ));
    }

    private Restriction createTimeRestrictions(AuditLogQuery query) {
        final List<Restriction> restrictions = new ArrayList<>(2);
        if (query.getOnOrAfter() != null) {
            restrictions.add(new SimpleRestriction(CHANGESET_ALIAS + "." + CHANGESET_TIMESTAMP_PROPERTY, ">=", query.getOnOrAfter().toEpochMilli()));
        }
        if (query.getBeforeOrOn() != null) {
            restrictions.add(new SimpleRestriction(CHANGESET_ALIAS + "." + CHANGESET_TIMESTAMP_PROPERTY, "<=", query.getBeforeOrOn().toEpochMilli()));
        }
        return createBooleanRestriction(BooleanRestriction.BooleanLogic.AND, restrictions);
    }

    private void appendFrom(HQLQuery hqlQuery, AuditLogQuery query) {
        final AuditLogChangesetProjection projection = query.getProjection();
        if (projection == null) {
            hqlQuery.appendSelect(CHANGESET_ALIAS);
        } else {
            hqlQuery.requireDistinct();
            hqlQuery.appendSelect(AuditLogQueryProjectionTranslator.selectFor(query));
            hqlQuery.setResultTransform(AuditLogQueryProjectionTranslator.resultMapperFor(query).andThen(Function.identity()));
        }

        hqlQuery.appendFrom(CHANGESET_ENTITY).append(" ").append(CHANGESET_ALIAS);
    }

    private <T, U extends Restriction> List<Restriction> appendBooleanLogicClauseForCollection(Collection<T> elements, Function<T, U> appender) {
        return elements.stream().map(appender).collect(Collectors.toList());
    }

    private Restriction createBooleanRestriction(BooleanRestriction.BooleanLogic booleanLogic, List<Restriction> restrictions) {
        if (restrictions.isEmpty()) {
            return new EmptyRestriction();
        } else {
            return new BooleanHqlRestriction(booleanLogic, restrictions);
        }
    }

    private Restriction createRestrictionGroup(List<Restriction> restrictions) {
        if (restrictions.isEmpty()) {
            return new EmptyRestriction();
        }
        return new RestrictionGroup(restrictions);
    }

    private Restriction createClauseWithJoin(List<AuditLogQueryEntityRestriction> entities, String join, String alias, AuditLogEntityType entityType) {
        if (entities.isEmpty()) {
            return new EmptyRestriction();
        } else {
            final ImmutableList.Builder<Restriction> restrictions = new ImmutableList.Builder<>();
            restrictions.add(new SimpleRestriction(alias + "." + ENTITY_TYPE_PROPERTY, "=", entityType));
            restrictions.add(new RestrictionGroup(
                    entities.stream()
                            .map(createEntityRestrictionMapper(alias + "." + ENTITY_ID_PROPERTY, alias + "." + ENTITY_NAME_PROPERTY))
                            .collect(Collectors.toList()))
            );
            return new RestrictionWithJoin(join, new BooleanHqlRestriction(BooleanRestriction.BooleanLogic.AND, restrictions.build()));
        }
    }

    private Function<AuditLogQueryEntityRestriction, Restriction> createEntityRestrictionMapper(String idProperty, String nameProperty) {
        return entity -> {
            final ImmutableList.Builder<Restriction> restrictions = buildNameIdRestrictions(idProperty, nameProperty, entity);
            return createBooleanRestriction(BooleanRestriction.BooleanLogic.AND, restrictions.build());
        };
    }

    private Function<AuditLogQueryAuthorRestriction, Restriction> createAuthorRestrictionMapper(String idProperty, String nameProperty) {
        return author -> {
            final ImmutableList.Builder<Restriction> restrictions = buildNameIdRestrictions(idProperty, nameProperty, author);

            if (author.getType() != null) {
                restrictions.add(new SimpleRestriction(CHANGESET_ALIAS + "." + CHANGESET_AUTHOR_TYPE_PROPERTY, "=", author.getType()));
            }
            return createBooleanRestriction(BooleanRestriction.BooleanLogic.AND, restrictions.build());
        };
    }

    private ImmutableList.Builder<Restriction> buildNameIdRestrictions(String idProperty,
                                                                       String nameProperty,
                                                                       AuditLogQueryEntityRestriction entity) {
        final ImmutableList.Builder<Restriction> restrictions = new ImmutableList.Builder<>();
        if (entity.getId() != null) {
            restrictions.add(new SimpleRestriction(idProperty, "=", entity.getId()));
        } else if (!Strings.isNullOrEmpty(entity.getName())) {
            restrictions.add(new SimpleRestriction(nameProperty, "=", entity.getName()));
        } else if (!Strings.isNullOrEmpty(entity.getNamePrefix())) {
            restrictions.add(new PrefixRestriction(nameProperty, entity.getNamePrefix()));
        }
        return restrictions;
    }
}
