package com.atlassian.crowd.search.hibernate.audit;

import com.atlassian.crowd.audit.AuditLogAuthorType;
import com.atlassian.crowd.audit.AuditLogEntity;
import com.atlassian.crowd.audit.AuditLogEntityType;
import com.atlassian.crowd.audit.AuditLogEventType;
import com.atlassian.crowd.audit.ImmutableAuditLogAuthor;
import com.atlassian.crowd.audit.ImmutableAuditLogEntity;
import com.atlassian.crowd.audit.query.AuditLogChangesetProjection;
import com.atlassian.crowd.audit.query.AuditLogQuery;
import com.google.common.base.Preconditions;

import java.util.Arrays;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.APPLICATION_ALIAS;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.AUTHOR_ID_PROPERTY;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.AUTHOR_NAME_PROPERTY;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.CHANGESET_ALIAS;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.CHANGESET_AUTHOR_TYPE_PROPERTY;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.CHANGESET_PROJECTION_KEY_ALIAS;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.DIRECTORY_ALIAS;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.ENTITY_ID_PROPERTY;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.ENTITY_NAME_PROPERTY;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.ENTITY_TYPE_PROPERTY;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.EVENT_TYPE_PROPERTY;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.GROUP_ALIAS;
import static com.atlassian.crowd.search.hibernate.audit.AuditLogQueryTranslator.USER_ALIAS;

abstract class AuditLogQueryProjectionTranslator {
    private static final Collector<CharSequence, ?, String> COLUMN_JOINER = Collectors.joining(", ");

    static String selectFor(AuditLogQuery<?> query) {
        final AuditLogChangesetProjection projection = Preconditions.checkNotNull(query.getProjection());
        switch (projection) {
            case EVENT_TYPE:
                return columns(CHANGESET_ALIAS, EVENT_TYPE_PROPERTY);
            case AUTHOR:
                return columns(CHANGESET_ALIAS, AUTHOR_NAME_PROPERTY, AUTHOR_ID_PROPERTY, CHANGESET_AUTHOR_TYPE_PROPERTY);
            case ENTITY_USER:
                Preconditions.checkState(!query.getUsers().isEmpty(), "Need to specify user entity restriction to do user entity projection");
                return entityColumns(USER_ALIAS);
            case ENTITY_GROUP:
                Preconditions.checkState(!query.getGroups().isEmpty(), "Need to specify group entity restriction to do group entity projection");
                return entityColumns(GROUP_ALIAS);
            case ENTITY_APPLICATION:
                Preconditions.checkState(!query.getApplications().isEmpty(), "Need to specify application entity restriction to do application entity projection");
                return entityColumns(APPLICATION_ALIAS);
            case ENTITY_DIRECTORY:
                Preconditions.checkState(!query.getDirectories().isEmpty(), "Need to specify directory entity restriction to do directory entity projection");
                return entityColumns(DIRECTORY_ALIAS);
            case SOURCE:
            default:
                throw new IllegalArgumentException("Unsupported projection " + projection);
        }
    }

    static <RESULT> Function<List<Object[]>, List<RESULT>> resultMapperFor(AuditLogQuery<RESULT> query) {
        final AuditLogChangesetProjection projection = Preconditions.checkNotNull(query.getProjection());
        switch (projection) {
            case EVENT_TYPE:
                checkProjectionReturnType(query, AuditLogEventType.class);
                return uniqueChangesets(row -> (RESULT) row[0]);
            case AUTHOR:
                checkProjectionReturnType(query, ImmutableAuditLogAuthor.class);
                return uniqueChangesets(row -> (RESULT) new ImmutableAuditLogAuthor((Long) row[1], (String) row[0], (AuditLogAuthorType) row[2]));
            case ENTITY_USER:
            case ENTITY_APPLICATION:
            case ENTITY_DIRECTORY:
            case ENTITY_GROUP:
                checkProjectionReturnType(query, AuditLogEntity.class);
                return uniqueChangesets(row -> {
                    final ImmutableAuditLogEntity.Builder entity = new ImmutableAuditLogEntity.Builder()
                            .setEntityId((Long) row[1])
                            .setEntityName((String) row[0])
                            .setEntityType((AuditLogEntityType) row[2]);
                    return (RESULT) entity.build();
                });
            case SOURCE:
                throw new IllegalArgumentException("Unsupported projection " + projection);
        }
        throw new IllegalArgumentException("Unsupported projection " + projection);
    }

    private static void checkProjectionReturnType(AuditLogQuery<?> query, final Class<?> expected) {
        final Class<?> returnType = query.getReturnType();
        final AuditLogChangesetProjection projection = query.getProjection();
        Preconditions.checkArgument(returnType.isAssignableFrom(expected), "Unsupported return type %s for projection %s", returnType, projection);
    }

    private static String entityColumns(final String entityAlias) {
        // intentionally coalescing primary and non-primary entities here
        return columns(entityAlias, ENTITY_NAME_PROPERTY, ENTITY_ID_PROPERTY, ENTITY_TYPE_PROPERTY);
    }

    private static String columns(final String alias, final String... columns) {
        return Stream.concat(
                Arrays.stream(columns).limit(1).map(column1 -> column1 + " as " + CHANGESET_PROJECTION_KEY_ALIAS),
                Arrays.stream(columns).skip(1)
        ).map(column -> alias + "." + column).collect(COLUMN_JOINER);
    }

    private static <RESULT> Function<List<Object[]>, List<RESULT>> uniqueChangesets(Function<Object[], RESULT> mapping) {
        return (rows -> rows.stream().map(mapping).collect(Collectors.toList()));
    }
}