package com.atlassian.crowd.search.hibernate;

import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

public class HQLQuery {
    protected final StringBuilder select = new StringBuilder("");
    protected final StringBuilder from = new StringBuilder(" FROM ");
    protected final StringBuilder where = new StringBuilder(" WHERE ");
    protected final StringBuilder orderBy = new StringBuilder(" ORDER BY ");
    protected int aliasCounter = 0;
    protected String cacheRegion;
    protected boolean distinctRequired = false;
    protected boolean whereRequired = false;
    protected boolean orderByRequired = false;
    protected boolean readOnly = false;
    protected final Map<String, Object> parameterValues = new LinkedHashMap<>();
    protected int maxResults;
    protected int startIndex;

    protected String batchedParam;
    protected Comparator comparatorForBatch;
    protected Function<List<Object[]>, List<?>> resultTransform;

    public StringBuilder appendSelect(CharSequence hql) {
        select.append(hql);
        return this.select;
    }

    public StringBuilder appendFrom(CharSequence hql) {
        from.append(hql);
        return this.from;
    }

    /**
     * @deprecated use safeAppendWhere instead
     * @param hql the text to append to the where clause
     * @return the stringbuilder representing the where clause
     */
    @Deprecated
    public StringBuilder appendWhere(CharSequence hql) {
        whereRequired = true;
        where.append(hql);
        return this.where;
    }

    public HQLQuery safeAppendWhere(CharSequence hql) {
        if (hql.toString().trim().length() > 0) {
            whereRequired = true;
            where.append(hql);
        }
        return this;
    }

    public StringBuilder appendOrderBy(CharSequence hql) {
        orderByRequired = true;
        orderBy.append(hql);
        return this.orderBy;
    }

    public int getNextAlias() {
        aliasCounter++;
        return aliasCounter;
    }

    /**
     * Creates a variable placeholder for a parameter
     *
     * @param value actual value of the parameter, or <code>null</code> if the parameter is to be ignored
     * @return the JPA-style named variable (:param1, :param2, ...)
     */
    public String addParameterPlaceholder(@Nullable Object value) {
        return ":" + addParam(value);
    }

    /**
     * Creates a variable placeholder for a parameter. Marks the parameter as batched; the name and the values
     * of the parameter may be retrieved using {@link #getBatchedParamName()} and {@link #getBatchedParamValues()}.
     * Only one parameter can be batched.
     *
     * @param values values of the parameter that need to be batched
     * @return the JPA-style named variable (:param1, :param2, ...)
     */
    public String addParameterPlaceholderForBatchedParam(@Nullable Collection values) {
        Preconditions.checkState(batchedParam == null, "Only one batched param allowed");
        this.batchedParam = addParam(values);
        return ":" + batchedParam;
    }

    /**
     * @return comparator to merge results of batched query
     */
    public Comparator getComparatorForBatch() {
        return comparatorForBatch;
    }

    /**
     * sets comparator to merge results of batched query
     */
    public void setComparatorForBatch(Comparator comparatorForBatch) {
        this.comparatorForBatch = comparatorForBatch;
    }

    private String addParam(@Nullable Object value) {
        String name = "param" + (parameterValues.size() + 1);
        parameterValues.put(name, value);
        return name;
    }

    /**
     * @return a list of parameter values. May contain <code>null</code> elements for parameters that are
     * to be ignored
     */
    public Map<String, Object> getParameterMap() {
        return Collections.unmodifiableMap(parameterValues);
    }

    public Collection<Object> getParameterValues() {
        return Collections.unmodifiableCollection(parameterValues.values());
    }

    public void requireDistinct() {
        distinctRequired = true;
    }

    public void limitResults(int maxResults) {
        this.maxResults = maxResults;
    }

    public void offsetResults(int startIndex) {
        this.startIndex = startIndex;
    }

    public int getMaxResults() {
        return maxResults;
    }

    public int getStartIndex() {
        return startIndex;
    }

    public boolean isReadOnly() {
        return readOnly;
    }

    public void setReadOnly(boolean readOnly) {
        this.readOnly = readOnly;
    }

    /**
     * Returns name of the multi-valued parameter that should be split into batches.
     */
    public String getBatchedParamName() {
        return batchedParam;
    }

    /**
     * Returns values of the multi-valued parameter that should be split into batches.
     */
    public Collection getBatchedParamValues() {
        return batchedParam == null ? null : (Collection) parameterValues.get(batchedParam);
    }

    public void setResultTransform(Function<List<Object[]>, List<?>> resultTransform) {
        Preconditions.checkState(this.resultTransform == null, "Result transform already set");
        this.resultTransform = resultTransform;
    }

    public Function<List<Object[]>, List<?>> getResultTransform() {
        return resultTransform;
    }

    public String getCacheRegion() {
        return cacheRegion;
    }

    public void setCacheRegion(String cacheRegion) {
        this.cacheRegion = cacheRegion;
    }

    @Override
    public String toString() {
        StringBuilder hql = new StringBuilder("SELECT ");
        if (distinctRequired) {
            hql.append("DISTINCT ");
        }

        hql.append(select);
        hql.append(from);

        if (whereRequired) {
            hql.append(where);
        }
        if (orderByRequired) {
            hql.append(orderBy);
        }
        return hql.toString();
    }
}
