package com.atlassian.crowd.search.hibernate;

import com.atlassian.crowd.model.InternalEntityAttribute;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.ImmutableDirectoryGroup;
import com.atlassian.crowd.model.group.ImmutableDirectoryGroupWithAttributes;
import com.atlassian.crowd.model.group.ImmutableGroup;
import com.atlassian.crowd.model.group.InternalGroup;
import com.atlassian.crowd.model.user.ImmutableTimestampedUser;
import com.atlassian.crowd.model.user.ImmutableTimestampedUserWithAttributes;
import com.atlassian.crowd.model.user.ImmutableUser;
import com.atlassian.crowd.model.user.InternalUser;
import com.atlassian.crowd.model.user.User;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Helper class providing {@link CustomDataFetcher}.
 */
public class CustomDataFetchers {
    private static final String DIRECTORY_ID = "directory.id";
    private static final String GROUP_TYPE = "type";
    private static final String ACTIVE = "active";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";
    private static final String DISPLAY_NAME = "displayName";
    private static final String EMAIL = "emailAddress";
    private static final String DESCRIPTION = "description";
    private static final String EXTERNAL_ID = "externalId";
    private static final String CREATED_DATE = "createdDate";
    private static final String UPDATED_DATE = "updatedDate";
    private static final String LOCAL = "local";

    private static final CustomDataFetcher<?> NO_OP = new CustomDataFetcher() {
        @Override
        public List<String> attributes(String alias) {
            return ImmutableList.of(alias);
        }

        @Override
        public Function<Object[], ?> getTransformer(int start) {
            return values -> values[start];
        }
    };

    private static final ImmutableMap<Class<? extends User>, CustomDataFetcher<?>> USER_PRODUCERS = ImmutableMap.of(
            ImmutableTimestampedUser.class, crateImmutableTimestampedUserProducer(),
            ImmutableUser.class, createImmutableUserProducer(),
            ImmutableTimestampedUserWithAttributes.class, createImmutableTimestampedUserWithAttributesProducer());

    private static final ImmutableMap<Class<? extends Group>, CustomDataFetcher<?>> GROUP_PRODUCERS = ImmutableMap.of(
            ImmutableDirectoryGroup.class, createImmutableDirectoryGroupProducer(),
            ImmutableGroup.class, createImmutableGroupProducer(),
            ImmutableDirectoryGroupWithAttributes.class, createImmutableDirectoryGroupWithAttributesProducer());

    private CustomDataFetchers() {
    }

    /**
     * Returns {@link CustomDataFetcher} appropriate for the given type.
     */
    public static <T> CustomDataFetcher<T> entityProducer(Class<T> returnType) {
        return Stream.of(USER_PRODUCERS, GROUP_PRODUCERS)
                .flatMap(e -> e.entrySet().stream())
                .filter(entry -> returnType.isAssignableFrom(entry.getKey()))
                .findFirst()
                .map(entry -> (CustomDataFetcher<T>) entry.getValue())
                .orElse((CustomDataFetcher<T>) NO_OP);
    }

    private static CustomDataFetcher<ImmutableTimestampedUserWithAttributes> createImmutableTimestampedUserWithAttributesProducer() {
        return new CustomDataFetcher<ImmutableTimestampedUserWithAttributes>() {
            @Override
            public List<String> attributes(String alias) {
                return ImmutableList.of(alias);
            }

            @Override
            public Function<Object[], ImmutableTimestampedUserWithAttributes> getTransformer(int start) {
                final IdentityHashMap<InternalUser, ImmutableTimestampedUserWithAttributes> cache = new IdentityHashMap<>();
                return values -> cache.computeIfAbsent((InternalUser) values[start], this::transform);
            }

            private ImmutableTimestampedUserWithAttributes transform(InternalUser user) {
                return ImmutableTimestampedUserWithAttributes.builder(
                        user, InternalEntityAttribute.toMap(user.getAttributes()))
                        .build();
            }
        };
    }

    private static CustomDataFetcher<ImmutableDirectoryGroupWithAttributes> createImmutableDirectoryGroupWithAttributesProducer() {
        return new CustomDataFetcher<ImmutableDirectoryGroupWithAttributes>() {
            @Override
            public List<String> attributes(String alias) {
                return ImmutableList.of(alias);
            }

            @Override
            public Function<Object[], ImmutableDirectoryGroupWithAttributes> getTransformer(int start) {
                final IdentityHashMap<InternalGroup, ImmutableDirectoryGroupWithAttributes> cache = new IdentityHashMap<>();
                return values -> cache.computeIfAbsent((InternalGroup) values[start], this::transform);
            }

            private ImmutableDirectoryGroupWithAttributes transform(InternalGroup group) {
                return ImmutableDirectoryGroupWithAttributes.builder(
                        group, InternalEntityAttribute.toMap(group.getAttributes()))
                        .build();
            }
        };
    }

    private static CustomDataFetcher<ImmutableDirectoryGroup> createImmutableDirectoryGroupProducer() {
        SetterBuilder<ImmutableDirectoryGroup.Builder> setters = new SetterBuilder<>();
        setters.put(DIRECTORY_ID, ImmutableDirectoryGroup.Builder::setDirectoryId);
        setters.put(GROUP_TYPE, ImmutableDirectoryGroup.Builder::setType);
        setters.put(ACTIVE, ImmutableDirectoryGroup.Builder::setActive);
        setters.put(DESCRIPTION, ImmutableDirectoryGroup.Builder::setDescription);
        setters.put(EXTERNAL_ID, ImmutableDirectoryGroup.Builder::setExternalId);
        setters.put(LOCAL, ImmutableDirectoryGroup.Builder::setLocal);

        setters.put(CREATED_DATE, ImmutableDirectoryGroup.Builder::setCreatedDate);
        setters.put(UPDATED_DATE, ImmutableDirectoryGroup.Builder::setUpdatedDate);
        return create(ImmutableDirectoryGroup::builder, ImmutableDirectoryGroup.Builder::build, setters);
    }

    private static CustomDataFetcher<ImmutableGroup> createImmutableGroupProducer() {
        SetterBuilder<ImmutableGroup.Builder> setters = new SetterBuilder<>();
        setters.put(DIRECTORY_ID, ImmutableGroup.Builder::setDirectoryId);
        setters.put(GROUP_TYPE, ImmutableGroup.Builder::setType);
        setters.put(ACTIVE, ImmutableGroup.Builder::setActive);
        setters.put(DESCRIPTION, ImmutableGroup.Builder::setDescription);
        setters.put(EXTERNAL_ID, ImmutableGroup.Builder::setExternalId);

        return create(ImmutableGroup::builder, ImmutableGroup.Builder::build, setters);
    }

    private static CustomDataFetcher<ImmutableTimestampedUser> crateImmutableTimestampedUserProducer() {
        SetterBuilder<ImmutableTimestampedUser.Builder> setters = new SetterBuilder<>();
        setters.put(DIRECTORY_ID, ImmutableTimestampedUser.Builder::directoryId);
        setters.put(DISPLAY_NAME, ImmutableTimestampedUser.Builder::displayName);
        setters.put(EMAIL, ImmutableTimestampedUser.Builder::emailAddress);
        setters.put(ACTIVE, ImmutableTimestampedUser.Builder::active);
        setters.put(FIRST_NAME, ImmutableTimestampedUser.Builder::firstName);
        setters.put(LAST_NAME, ImmutableTimestampedUser.Builder::lastName);
        setters.put(EXTERNAL_ID, ImmutableTimestampedUser.Builder::externalId);
        setters.put(CREATED_DATE, ImmutableTimestampedUser.Builder::createdDate);
        setters.put(UPDATED_DATE, ImmutableTimestampedUser.Builder::updatedDate);
        return create(ImmutableTimestampedUser::builder, ImmutableTimestampedUser.Builder::build, setters);
    }

    private static CustomDataFetcher<ImmutableUser> createImmutableUserProducer() {
        SetterBuilder<ImmutableUser.Builder> setters = new SetterBuilder<>();
        setters.put(DIRECTORY_ID, ImmutableUser.Builder::directoryId);
        setters.put(DISPLAY_NAME, ImmutableUser.Builder::displayName);
        setters.put(EMAIL, ImmutableUser.Builder::emailAddress);
        setters.put(ACTIVE, ImmutableUser.Builder::active);
        setters.put(FIRST_NAME, ImmutableUser.Builder::firstName);
        setters.put(LAST_NAME, ImmutableUser.Builder::lastName);
        setters.put(EXTERNAL_ID, ImmutableUser.Builder::externalId);
        return create(ImmutableUser::builder, ImmutableUser.Builder::build, setters);
    }

    private static class SetterBuilder<T> {
        Map<String, BiConsumer<T, Object>> setters = new LinkedHashMap<>();

        <Q> void put(String property, BiConsumer<T, Q> consumer) {
            setters.put(property, (entity, value) -> consumer.accept(entity, (Q) value));
        }
    }

    private static <T, Q> CustomDataFetcher<Q> create(Function<String, T> fromNameBuilder,
                                                      Function<T, Q> buildSupplier,
                                                      SetterBuilder<T> setterBuilder) {
        // ImmutableMap has stable iteration order
        final Map<String, BiConsumer<T, Object>> immutableSetters = ImmutableMap.copyOf(setterBuilder.setters);
        final List<String> attributes = ImmutableList.<String>builder()
                .add("id")
                .add("name")
                .addAll(immutableSetters.keySet())
                .build();

        return new CustomDataFetcher<Q>() {
            @Override
            public List<String> attributes(String alias) {
                return attributes.stream().map(att -> alias + "." + att).collect(Collectors.toList());
            }

            @Override
            public Function<Object[], Q> getTransformer(int start) {
                final HashMap<Object, Q> cache = new HashMap<>();
                return values -> cache.computeIfAbsent(values[start], id -> transform(values, start + 1));
            }

            private Q transform(Object[] values, int start) {
                int idx = start;
                T builder = fromNameBuilder.apply((String) values[idx++]);
                for (BiConsumer<T, Object> setter : immutableSetters.values()) {
                    setter.accept(builder, values[idx++]);
                }
                return buildSupplier.apply(builder);
            }
        };
    }
}
