package com.atlassian.crowd.model.membership;

import java.io.Serializable;
import java.util.Date;

import javax.annotation.Nullable;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.directory.DirectoryImpl;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.InternalGroup;
import com.atlassian.crowd.model.user.InternalUser;

import com.atlassian.crowd.model.user.MinimalUser;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.builder.ToStringBuilder;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;

/**
 * Encapsulates the concept of membership.
 */
public class InternalMembership implements Serializable {
    private Long id;

    private Long parentId;
    private Long childId;
    private MembershipType membershipType;

    protected Date createdDate;

    // redundant fields for optimisation
    private String parentName;
    private String lowerParentName;
    private String childName;
    private String lowerChildName;
    private GroupType groupType;
    private Directory directory;

    protected InternalMembership() {
    }

    /**
     * This constructor is only used for XML imports.
     */
    public InternalMembership(final Long id, final Long parentId, final Long childId,
                              final MembershipType membershipType, GroupType groupType,
                              final String parentName, final String childName, final Directory directory,
                              @Nullable Date createdDate) {
        this.id = id;
        this.parentId = parentId;
        this.childId = childId;
        this.membershipType = membershipType;
        this.groupType = groupType;
        setParentName(parentName);
        setChildName(childName);
        this.createdDate = createdDate;
        this.directory = directory;
    }

    private InternalMembership(
            InternalGroup parent, DirectoryEntity child, Date createdDate, long childId, MembershipType type) {
        Validate.notNull(parent, "group argument cannot be null");
        Validate.notNull(child, "child argument cannot be null");
        Validate.isTrue(parent.getDirectoryId() == child.getDirectoryId(), "directoryIDs of the parent and child do not match");

        this.parentId = parent.getId();
        this.childId = childId;
        this.membershipType = type;
        this.groupType = parent.getType();
        setParentName(parent.getName());
        setChildName(child.getName());
        this.createdDate = createdDate;
        this.directory = parent.getDirectory();
    }


    public InternalMembership(InternalGroup parentGroup, InternalUser user, Date createdDate) {
        this(parentGroup, user, createdDate, user.getId(), MembershipType.GROUP_USER);
    }

    public InternalMembership(InternalGroup parentGroup, MinimalUser user, Date createdDate) {
        this(parentGroup, user, createdDate, user.getId(), MembershipType.GROUP_USER);
    }

    public InternalMembership(InternalGroup parentGroup, InternalGroup childGroup, Date createdDate) {
        this(parentGroup, childGroup, createdDate, childGroup.getId(), MembershipType.GROUP_GROUP);
        Validate.isTrue(parentGroup.getType().equals(childGroup.getType()), "groupTypes of the parent and child group do not match");
    }

    public Long getId() {
        return id;
    }

    private void setId(final Long id) {
        this.id = id;
    }

    public Long getParentId() {
        return parentId;
    }

    public Long getChildId() {
        return childId;
    }

    public MembershipType getMembershipType() {
        return membershipType;
    }

    public String getParentName() {
        return parentName;
    }

    public String getChildName() {
        return childName;
    }

    public Directory getDirectory() {
        return directory;
    }

    public GroupType getGroupType() {
        return groupType;
    }

    public String getLowerParentName() {
        return lowerParentName;
    }

    public String getLowerChildName() {
        return lowerChildName;
    }

    private void setParentId(final Long parentId) {
        this.parentId = parentId;
    }

    private void setChildId(final Long childId) {
        this.childId = childId;
    }

    private void setMembershipType(final MembershipType membershipType) {
        this.membershipType = membershipType;
    }

    private void setParentName(final String parentName) {
        this.parentName = parentName;
        this.lowerParentName = toLowerCase(parentName);
    }

    private void setChildName(final String childName) {
        this.childName = childName;
        this.lowerChildName = toLowerCase(childName);
    }

    private void setDirectory(final DirectoryImpl directory) {
        this.directory = directory;
    }

    private void setGroupType(final GroupType groupType) {
        this.groupType = groupType;
    }

    private void setLowerParentName(final String lowerParentName) {
        this.lowerParentName = lowerParentName;
    }

    private void setLowerChildName(final String lowerChildName) {
        this.lowerChildName = lowerChildName;
    }

    public Date getCreatedDate() {
        return createdDate;
    }

    protected void setCreatedDate(Date createdDate) {
        this.createdDate = createdDate;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof InternalMembership)) {
            return false;
        }

        InternalMembership that = (InternalMembership) o;

        if (getChildId() != null ? !getChildId().equals(that.getChildId()) : that.getChildId() != null) {
            return false;
        }
        if (getParentId() != null ? !getParentId().equals(that.getParentId()) : that.getParentId() != null) {
            return false;
        }
        if (getMembershipType() != that.getMembershipType()) {
            return false;
        }

        return true;
    }

    @Override
    public int hashCode() {
        int result = getParentId() != null ? getParentId().hashCode() : 0;
        result = 31 * result + (getChildId() != null ? getChildId().hashCode() : 0);
        result = 31 * result + (getMembershipType() != null ? getMembershipType().hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this).
                append("parentId", getParentId()).
                append("childId", getChildId()).
                append("membershipType", getMembershipType()).
                append("groupType", getGroupType()).
                append("parentName", getParentName()).
                append("lowerParentName", getLowerParentName()).
                append("childName", getChildName()).
                append("lowerChildName", getLowerChildName()).
                append("directoryId", getDirectory().getId()).
                toString();
    }
}
