package com.atlassian.crowd.model.group;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.model.InternalDirectoryEntity;
import com.atlassian.crowd.model.InternalEntityTemplate;
import com.atlassian.crowd.model.permission.GroupAdministrationGrantToGroup;
import com.atlassian.crowd.model.permission.UserAdministrationGrantToGroup;
import com.atlassian.crowd.util.InternalEntityUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.builder.ToStringBuilder;

import javax.annotation.Nullable;
import java.util.HashSet;
import java.util.Set;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;

/**
 * Encapsulates the concept of group.
 */
public class InternalGroup extends InternalDirectoryEntity<InternalGroupAttribute> implements InternalDirectoryGroup {
    private String lowerName;
    private GroupType type;
    private String description;
    private boolean isLocal;
    private String externalId;
    private Set<GroupAdministrationGrantToGroup> grantsToOtherGroups = new HashSet<>();
    private Set<GroupAdministrationGrantToGroup> groupGrantsToThisGroup = new HashSet<>();
    private Set<UserAdministrationGrantToGroup> userGrantsToThisGroup = new HashSet<>();

    protected InternalGroup() {

    }

    // this constructor is used by the importer
    public InternalGroup(final InternalEntityTemplate internalEntityTemplate, final Directory directory, final GroupTemplate groupTemplate) {
        super(internalEntityTemplate, directory);

        this.type = groupTemplate.getType();

        updateDetailsFrom(groupTemplate);
    }

    // constructor for new groups
    public InternalGroup(final Group group, final Directory directory) {
        super();

        Validate.notNull(directory, "directory argument cannot be null");

        setName(group.getName());
        this.directory = directory;
        this.type = group.getType();
        this.externalId = group.getExternalId();

        updateDetailsFrom(group);
    }

    // MUTATOR
    private void validateGroup(final Group group) {
        Validate.notNull(group, "group argument cannot be null");
        Validate.notNull(group.getDirectoryId(), "group argument cannot have a null directoryID");
        Validate.notNull(group.getName(), "group argument cannot have a null name");
        Validate.notNull(group.getType(), "type argument cannot be null");

        Validate.isTrue(group.getDirectoryId() == this.getDirectoryId(), "directoryID of updated group does not match the directoryID of the existing group.");
        Validate.isTrue(group.getName().equals(this.getName()), "group name of updated group does not match the group name of the existing group.");
    }

    // MUTATOR
    public void updateDetailsFrom(final Group group) {
        validateGroup(group);

        this.active = group.isActive();
        this.description = InternalEntityUtils.truncateValue(group.getDescription());
        this.externalId = group.getExternalId();
        // group type is not updated
    }

    // MUTATOR
    public void renameTo(String newName) {
        Validate.isTrue(StringUtils.isNotBlank(newName), "the new name cannot be null or blank");

        setName(newName);
    }

    @Override
    protected void setName(final String name) {
        InternalEntityUtils.validateLength(name);
        this.name = name;
        this.lowerName = toLowerCase(name);
    }

    public String getDescription() {
        return description;
    }

    public GroupType getType() {
        return type;
    }

    public String getLowerName() {
        return lowerName;
    }

    private void setLowerName(final String lowerName) {
        this.lowerName = lowerName;
    }

    private void setDescription(final String description) {
        this.description = description;
    }

    private void setType(final GroupType type) {
        this.type = type;
    }

    public boolean isLocal() {
        return isLocal;
    }

    public void setLocal(final boolean local) {
        isLocal = local;
    }

    // mapped in Hibernate but not used in our code
    @SuppressWarnings("UnusedDeclaration")
    private void setAttributes(final Set<InternalGroupAttribute> attributes) {
        this.attributes = attributes;
    }

    @Nullable
    public String getExternalId() {
        return externalId;
    }

    public void setExternalId(String externalId) {
        this.externalId = externalId;
    }

    private Set<GroupAdministrationGrantToGroup> getGrantsToOtherGroups() {
        return grantsToOtherGroups;
    }

    private void setGrantsToOtherGroups(Set<GroupAdministrationGrantToGroup> grantsToOtherGroups) {
        this.grantsToOtherGroups = grantsToOtherGroups;
    }

    private Set<GroupAdministrationGrantToGroup> getGroupGrantsToThisGroup() {
        return groupGrantsToThisGroup;
    }

    private void setGroupGrantsToThisGroup(Set<GroupAdministrationGrantToGroup> groupGrantsToThisGroup) {
        this.groupGrantsToThisGroup = groupGrantsToThisGroup;
    }

    private Set<UserAdministrationGrantToGroup> getUserGrantsToThisGroup() {
        return userGrantsToThisGroup;
    }

    private void setUserGrantsToThisGroup(Set<UserAdministrationGrantToGroup> userGrantsToThisGroup) {
        this.userGrantsToThisGroup = userGrantsToThisGroup;
    }

    @Override
    public boolean equals(final Object o) {
        return GroupComparator.equalsObject(this, o);
    }

    @Override
    public int hashCode() {
        return GroupComparator.hashCode(this);
    }

    public int compareTo(Group o) {
        return GroupComparator.compareTo(this, o);
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this).
                append("id", getId()).
                append("name", getName()).
                append("type", getType()).
                append("active", isActive()).
                append("description", getDescription()).
                append("lowerName", getLowerName()).
                append("createdDate", getCreatedDate()).
                append("updatedDate", getUpdatedDate()).
                append("directoryId", getDirectoryId()).
                append("externalId", getExternalId()).
                toString();
    }
}

