package com.atlassian.crowd.model.audit;

import com.atlassian.crowd.audit.AuditLogAuthor;
import com.atlassian.crowd.audit.AuditLogAuthorType;
import com.atlassian.crowd.audit.AuditLogChangeset;
import com.atlassian.crowd.audit.AuditLogEventSource;
import com.atlassian.crowd.audit.AuditLogEventType;
import com.atlassian.crowd.audit.ImmutableAuditLogAuthor;
import com.google.common.base.MoreObjects;

import java.time.Instant;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import static com.google.common.base.Strings.nullToEmpty;

public class AuditLogChangesetEntity implements AuditLogChangeset {
    private Long id;
    private Long timestamp;
    private AuditLogAuthorType authorType;
    private Long authorId;
    private String authorName = "";
    private AuditLogEventType eventType;
    private String ipAddress = "";
    private String eventMessage = "";
    private AuditLogEventSource source;
    private Set<AuditLogEntryEntity> entries = new HashSet<>();
    private Set<AuditLogEntityEntity> entities = new HashSet<>();

    public AuditLogChangesetEntity() {
    }

    public AuditLogChangesetEntity(AuditLogChangeset changeset) {
        this.id = changeset.getId();
        this.timestamp = changeset.getTimestampInstant().toEpochMilli();
        this.authorType = changeset.getAuthor().getType();
        this.authorId = changeset.getAuthor().getId();
        this.authorName = nullToEmpty(changeset.getAuthor().getName());
        this.eventType = changeset.getEventType();
        this.ipAddress = nullToEmpty(changeset.getIpAddress());
        this.eventMessage = nullToEmpty(changeset.getEventMessage());
        this.source = changeset.getSource();
        changeset.getEntities()
                .stream()
                .map((AuditLogEntityEntity::new))
                .forEach(entity -> {
                    entities.add(entity);
                    entity.setChangeset(this);
                });
        changeset.getEntries()
                .stream()
                .map(AuditLogEntryEntity::new)
                .forEach(auditLogEntry -> {
                    entries.add(auditLogEntry);
                    auditLogEntry.setChangeset(this);
                });
    }

    public Long getId() {
        return id;
    }

    @Override
    public Instant getTimestampInstant() {
        return timestamp != null ? Instant.ofEpochMilli(timestamp) : null;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public Long getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(Long timestamp) {
        this.timestamp = timestamp;
    }

    @Override
    public AuditLogAuthorType getAuthorType() {
        return authorType;
    }

    public void setAuthorType(AuditLogAuthorType authorType) {
        this.authorType = authorType;
    }

    @Override
    public Long getAuthorId() {
        return authorId;
    }

    public void setAuthorId(Long authorId) {
        this.authorId = authorId;
    }

    @Override
    public String getAuthorName() {
        return authorName;
    }

    public void setAuthorName(String authorName) {
        this.authorName = nullToEmpty(authorName);
    }

    @Override
    public AuditLogEventType getEventType() {
        return eventType;
    }

    public void setEventType(AuditLogEventType eventType) {
        this.eventType = eventType;
    }

    @Override
    public String getIpAddress() {
        return ipAddress;
    }

    public void setIpAddress(String ipAddress) {
        this.ipAddress = nullToEmpty(ipAddress);
    }

    @Override
    public String getEventMessage() {
        return eventMessage;
    }

    public void setEventMessage(String eventMessage) {
        this.eventMessage = nullToEmpty(eventMessage);
    }

    @Override
    public AuditLogEventSource getSource() {
        return source;
    }

    public void setSource(AuditLogEventSource source) {
        this.source = source;
    }

    public Set<AuditLogEntryEntity> getEntries() {
        return entries;
    }

    public void setEntries(Set<AuditLogEntryEntity> auditLogEntries) {
        this.entries = auditLogEntries;
    }

    @Override
    public Set<AuditLogEntityEntity> getEntities() {
        return entities;
    }

    public void setEntities(Set<AuditLogEntityEntity> entities) {
        this.entities = entities;
    }

    @Override
    public AuditLogAuthor getAuthor() {
        return new ImmutableAuditLogAuthor(authorId, authorName, authorType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        final AuditLogChangesetEntity that = (AuditLogChangesetEntity) o;
        return Objects.equals(this.getId(), that.getId())
                && Objects.equals(this.getTimestamp(), that.getTimestamp())
                && Objects.equals(this.getAuthorType(), that.getAuthorType())
                && Objects.equals(this.getAuthorId(), that.getAuthorId())
                && Objects.equals(this.getAuthorName(), that.getAuthorName())
                && Objects.equals(this.getEventType(), that.getEventType())
                && Objects.equals(this.getIpAddress(), that.getIpAddress())
                && Objects.equals(this.getEventMessage(), that.getEventMessage())
                && Objects.equals(this.getSource(), that.getSource())
                && Objects.equals(this.getEntries(), that.getEntries())
                && Objects.equals(this.getEntities(), that.getEntities());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getId(), getTimestamp(), getAuthorType(), getAuthorId(), getAuthorName(), getEventType(),
                getIpAddress(), getEventMessage(), getSource(), getEntries(), getEntities());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("id", id)
                .add("timestamp", timestamp)
                .add("authorType", authorType)
                .add("authorId", authorId)
                .add("authorName", authorName)
                .add("eventType", eventType)
                .add("ipAddress", ipAddress)
                .add("eventMessage", eventMessage)
                .add("entries", entries)
                .add("entities", entities)
                .add("source", source)
                .toString();
    }
}
