package com.atlassian.crowd.dao.rememberme;

import com.atlassian.crowd.model.rememberme.CrowdRememberMeToken;
import com.atlassian.crowd.model.rememberme.InternalCrowdRememberMeToken;

import javax.annotation.Nonnull;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

/**
 * Component representing database access layer for {@link InternalCrowdRememberMeToken}
 */
public interface CrowdRememberMeTokenDAO {

    /**
     * Find token by the id, if present
     */
    Optional<InternalCrowdRememberMeToken> findByIdExclusiveLock(Long id);

    /**
     * Lists tokens for the given username and the directory id
     * @param directoryId directory id to be searched for
     * @param username username of the user
     * @return collection of tokens
     */
    List<InternalCrowdRememberMeToken> findTokensForUser(String username, long directoryId);

    /**
     * Saves the token
     */
    InternalCrowdRememberMeToken save(InternalCrowdRememberMeToken token);

    /**
     * Removes all tokens having createdTime before the lastValidTokenCreateTime, or usedTime before the lastValidUsedBeforeTime
     *
     * @param lastValidTokenCreateTime Last valid token create timestamp (exclusive)
     * @param lastValidUsedBeforeTime  Last valid token used timestamp (exclusive)
     * @return number of tokens deleted
     */
    int removeAllExpiredTokens(LocalDateTime lastValidTokenCreateTime, LocalDateTime lastValidUsedBeforeTime);

    /**
     * Removes tokens for all the given series
     *
     * @param series Series to remove all tokens for
     * @return number of tokens deleted
     */
    int removeTokensForSeries(String series);

    /**
     * @param series Series to match token
     * @param token  token value to match
     * @return a matching remember me token if exists
     */
    Optional<InternalCrowdRememberMeToken> findBySeriesAndToken(@Nonnull String series, @Nonnull String token);

    /**
     * Removes all tokens for a user in the given directory
     *
     * @param username    username of the user
     * @param directoryId directory id to which the user belongs
     * @return number of tokens deleted
     */
    int removeTokensForUserInDirectory(String username, long directoryId);

    /**
     * Removes all the tokens for the given directory
     *
     * @param directoryId directory id to which the tokens belongs
     * @return number of tokens deleted
     */
    int removeTokensForDirectory(long directoryId);

    /**
     * Tries to claim the given token if it is not yet claimed (used)
     *
     * @param crowdRememberMeToken token to be claimed
     * @return whether attempt was successful or not
     */
    boolean setUsedByRemoteAddress(CrowdRememberMeToken crowdRememberMeToken);

    /**
     * Removes all remember me tokens present
     */
    void removeAll();

    /**
     * Reload the properties from database
     * @param token token to be refreshed
     */
    void refresh(InternalCrowdRememberMeToken token);

}
