package com.atlassian.crowd.dao.direntity;

import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.util.cache.LocalCacheUtils;

import javax.annotation.Nullable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledExecutorService;

/**
 * Implementation of {@link DirectoryEntityResolver} with local in-memory cache.
 */
public class LocallyCachedDirectoryEntityResolver implements DirectoryEntityResolver {
    private final ConcurrentMap<Object, ConcurrentMap<String, Object>> caches;
    private final Duration cacheTtl;

    public LocallyCachedDirectoryEntityResolver(Duration cacheTtl, ScheduledExecutorService cleanupPool) {
        this.caches = LocalCacheUtils.createExpiringAfterAccessMap(cacheTtl, cleanupPool);
        this.cacheTtl = cacheTtl;
    }

    @Nullable
    @Override
    public <T extends DirectoryEntity> T resolve(long directoryId, String name, Class<T> entityClass) {
        return resolve(getCache(directoryId, entityClass), name, entityClass);
    }

    @Override
    public <T extends DirectoryEntity> void put(T entity) {
        ConcurrentMap<String, Object> cache = getCache(entity.getDirectoryId(), entity.getClass());
        cache.put(IdentifierUtils.toLowerCase(entity.getName()), entity);
    }

    @Nullable
    @Override
    public <T extends DirectoryEntity> List<T> resolveAllOrNothing(long directoryId, Collection<String> names, Class<T> entityClass) {
        Map<String, Object> cache = getCache(directoryId, entityClass);
        List<T> results = new ArrayList<>();
        for (String name : names) {
            T t = resolve(cache, name, entityClass);
            if (t == null) {
                return null;
            }
            results.add(t);
        }
        return results;
    }

    <T> T resolve(Map<String, Object> cache, String name, Class<T> entityClass) {
        Object entity = cache.get(IdentifierUtils.toLowerCase(name));
        return entityClass.isInstance(entity) ? entityClass.cast(entity) : null;
    }

    @Override
    public void putAll(List<? extends DirectoryEntity> entities) {
        entities.forEach(this::put);
    }

    private ConcurrentMap<String, Object> getCache(long directoryId, Class<?> entityClass) {
        Object key = Arrays.asList(directoryId, User.class.isAssignableFrom(entityClass));
        return caches.computeIfAbsent(key, k -> LocalCacheUtils.createExpiringAfterAccessMap(cacheTtl));
    }
}
