package com.atlassian.crowd.crypto;

import com.atlassian.crowd.embedded.api.Encryptor;
import com.atlassian.crowd.model.property.Property;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.SetMultimap;

import java.util.function.UnaryOperator;

public class PropertyEncryptor {
    private Encryptor encryptor;
    private SetMultimap<String, String> encryptedNames;

    public PropertyEncryptor(Encryptor encryptor) {
        this(encryptor, ImmutableSetMultimap.of(Property.CROWD_PROPERTY_KEY, Property.MAILSERVER_PASSWORD));
    }

    /**
     * @param encryptor      encryptor to encrypt sensitive data
     * @param encryptedNames property key to set of property names to encrypt
     */
    public PropertyEncryptor(Encryptor encryptor, SetMultimap<String, String> encryptedNames) {
        this.encryptor = encryptor;
        this.encryptedNames = ImmutableSetMultimap.copyOf(encryptedNames);
    }

    public Property encrypt(Property property) {
        return transform(property, encryptor::encrypt);
    }

    public Property decrypt(Property property) {
        return transform(property, encryptor::decrypt);
    }

    public Property transform(Property property, UnaryOperator<String> transformer) {
        return shouldEncrypt(property)
                ? new Property(property.getKey(), property.getName(), transformer.apply(property.getValue()))
                : property;
    }

    private boolean shouldEncrypt(Property property) {
        return !Strings.isNullOrEmpty(property.getValue())
                && encryptedNames.get(property.getKey()).contains(property.getName());
    }
}
