package com.atlassian.crowd.crypto;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DataReEncryptor;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.stream.Collectors;

import static com.atlassian.crowd.search.query.DirectoryQueries.allDirectories;

public class EncryptingDirectoryDAO implements DirectoryDao, DataReEncryptor {
    private static final Logger log = LoggerFactory.getLogger(EncryptingDirectoryDAO.class);

    private DirectoryDao delegate;
    private DirectoryPasswordsEncryptor directoryPasswordsEncryptor;

    public void setDelegate(DirectoryDao delegate) {
        this.delegate = delegate;
    }

    public void setDirectoryPasswordsEncryptor(DirectoryPasswordsEncryptor directoryPasswordsEncryptor) {
        this.directoryPasswordsEncryptor = directoryPasswordsEncryptor;
    }

    @Override
    public Directory findById(long directoryId) throws DirectoryNotFoundException {
        return directoryPasswordsEncryptor.decryptPasswords(delegate.findById(directoryId));
    }

    @Override
    public Directory findByName(String name) throws DirectoryNotFoundException {
        return directoryPasswordsEncryptor.decryptPasswords(delegate.findByName(name));
    }

    @Override
    public List<Directory> findAll() {
        return delegate.findAll().stream()
                .map(directoryPasswordsEncryptor::decryptPasswords)
                .collect(Collectors.toList());
    }

    @Override
    public Directory add(Directory directory) {
        Directory encryptedDirectory = directoryPasswordsEncryptor.encryptPasswords(directory);
        Directory added = delegate.add(encryptedDirectory);
        return directoryPasswordsEncryptor.decryptPasswords(added);
    }

    @Override
    public Directory update(Directory directory) throws DirectoryNotFoundException {
        Directory encryptedDirectory = directoryPasswordsEncryptor.encryptPasswords(directory);
        Directory updated = delegate.update(encryptedDirectory);
        return directoryPasswordsEncryptor.decryptPasswords(updated);
    }

    @Override
    public void remove(Directory directory) throws DirectoryNotFoundException {
        delegate.remove(directory);
    }

    @Override
    public List<Directory> search(EntityQuery<Directory> entityQuery) {
        return delegate.search(entityQuery).stream()
                .map(directoryPasswordsEncryptor::decryptPasswords)
                .collect(Collectors.toList());
    }

    @Override
    public void reEncrypt() {
        for (Directory directory : search(allDirectories())) {
            try {
                update(directory);
            } catch (DirectoryNotFoundException e) {
                log.warn("Could not encrypt passwords of directory {}.", directory.getId(), e);
            } catch (Exception e) {
                log.error("Encryption of directory {} password failed.", directory.getId(), e);
            }
        }
    }
}
