package com.atlassian.crowd.integration.seraph;

import com.atlassian.crowd.embedded.api.CrowdService;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.integration.AuthenticationState;
import com.atlassian.crowd.integration.http.CacheAwareCrowdHttpAuthenticator;
import com.atlassian.crowd.integration.http.CrowdHttpAuthenticator;
import com.atlassian.crowd.service.AuthenticatorUserCache;
import com.atlassian.seraph.auth.AuthenticatorException;
import com.atlassian.seraph.auth.DefaultAuthenticator;
import com.atlassian.seraph.auth.LoginReason;
import com.atlassian.seraph.elevatedsecurity.ElevatedSecurityGuard;
import com.atlassian.seraph.filter.BaseLoginFilter;
import com.atlassian.seraph.util.RedirectUtils;
import com.google.common.base.Optional;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.util.concurrent.UncheckedExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.security.Principal;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import static com.atlassian.crowd.integration.AuthenticationState.authenticated;
import static com.atlassian.crowd.integration.AuthenticationState.unauthenticated;
import static com.atlassian.seraph.auth.LoginReason.OK;

public abstract class CrowdAuthenticator extends DefaultAuthenticator {
    private static final String SESSION_TOKEN_KEY = CrowdAuthenticator.class.getName() + "#SESSION_TOKEN_KEY";

    public static final String PASSWORD_RESET_REQUIRED_HEADER = "X-Seraph-PasswordResetRequired";

    protected static final Logger logger = LoggerFactory.getLogger(CrowdAuthenticator.class);

    private static final String CORRECT_PASSWORD = "c";
    private static final String INCORRECT_PASSWORD = "i";

    /**
     * If set to true will disable triggering the userAuthenticated() notifications on new SSO logins, which matches the behavior before 2.10.4/3.3.0
     */
    private static final Boolean DISABLE_USER_AUTHENTICATED_NOTIFICATIONS = Boolean.valueOf("crowd.integration.seraph.user.authenticated.notification.skip");

    /**
     * The number of seconds to avoid re-sending the userAuthenticated() notifications for a given username's SSO session.
     * This is to avoid multiple notifications if multiple requests come in at the same time without a local session, but with a SSO session.
     */
    private static final Integer USER_AUTHENTICATED_REFRESH_EXPIRATION_SECONDS = Integer.getInteger("crowd.integration.seraph.user.authenticated.notification.expiration", 60);

    private final Cache<String, Optional<String>> userAuthenticatedSentCache = CacheBuilder.newBuilder()
            .expireAfterWrite(USER_AUTHENTICATED_REFRESH_EXPIRATION_SECONDS, TimeUnit.SECONDS)
            .build();

    private final CrowdHttpAuthenticator crowdHttpAuthenticator;
    private final Supplier<CrowdService> crowdServiceSupplier;

    public CrowdAuthenticator(CrowdHttpAuthenticator crowdHttpAuthenticator, Supplier<CrowdService> crowdServiceSupplier) {
        this.crowdServiceSupplier = crowdServiceSupplier;
        this.crowdHttpAuthenticator = new CacheAwareCrowdHttpAuthenticator(crowdHttpAuthenticator, new AuthenticatorUserCache() {
            // CacheAwareCrowdHttpAuthenticator needs to know how to fetch users
            @Override
            public void fetchInCache(String username)
                    throws UserNotFoundException, InvalidAuthenticationException, OperationFailedException {
                fetchUserInCache(username);
            }
        });
    }

    /**
     * Fetches a user with the given username in the cache, in case the user
     * exists, but cannot be found from the cache yet.
     *
     * By default this method will call {@link #getUser(String)}, but JIRA needs
     * to override it, because {@link com.atlassian.seraph.auth.DefaultAuthenticator#getUser(String)}
     * only checks the local cache when retrieving users.
     *
     * @param username username of the user to be fetched
     * @throws com.atlassian.crowd.exception.InvalidAuthenticationException if the application or user authentication was not successful.
     * @throws com.atlassian.crowd.exception.OperationFailedException       if the operation has failed for an unknown reason
     */
    protected void fetchUserInCache(String username)
            throws UserNotFoundException, InvalidAuthenticationException, OperationFailedException {
        getUser(username);
    }

    /**
     * Override the super method, always return true so that authentication is not called twice when a user logs in.
     *
     * More info: this is because we subclass login() to perform the authentication, but also call super.login(),
     * which then calls this authenticate() method. We also can't just implement the authenticate() method as it
     * does not provide the HttpServletRequest nor the HttpServletResponse, which are both required for generating
     * and setting the Crowd SSO token.
     */
    @Override
    protected boolean authenticate(Principal user, String password) {
        return CORRECT_PASSWORD.equals(password);
    }

    /**
     * We must override the login() method as it gives us access to the HttpServletRequest and HttpServletResponse,
     * which Crowd needs in order to generate and set the Crowd SSO token.
     *
     * However, super.login() does some magic, including elevated security checks, so we still need to call
     * super.login() - which in turn calls authenticate().
     *
     * Problem is, we can't put our actual authentication login in their as authenticate() doesn't pass the
     * HttpServletRequest or HttpServletResponse into the method.
     *
     * Perhaps in a later version of Seraph, we can change authenticate to take the HttpServletRequest and
     * HttpServletResponse as parameters. But for now, we have a hacky solution that piggybacks the password
     * parameter so authenticate() knows whether to return true or false.
     *
     * @param request  HttpServletRequest obtain validation factors.
     * @param response HttpServletResponse SSO cookie is set on response.
     * @param username name of user to authenticate.
     * @param password credential to authenticate.
     * @param cookie   whether to set a remember-me cookie or not.
     * @return <code>true</code> if and only if authentication was successful
     */
    @Override
    public boolean login(HttpServletRequest request, HttpServletResponse response,
                         String username, String password, boolean cookie) throws AuthenticatorException {
        // as this is an explicit call to log in, we should ignore any previous authentication
        // if all we wanted to do was to check authentication, then we should call isAuthenticated() instead

        boolean authenticated;

        try {
            // clear anything that resembles a previous authentication
            this.logout(request, response);

            // unfortunately that will stamp OUT as the LoginReason, so we need to reset it
            request.setAttribute(LoginReason.REQUEST_ATTR_NAME, null);

            // run a clean new authentication
            // use the CrowdService api first, to make sure that the user is copied over/renamed/updated as needed
            logger.debug("Authenticating user with Crowd");
            crowdServiceSupplier.get().authenticate(username, password);

            logger.debug("Establishing SSO session");
            crowdHttpAuthenticator.authenticate(request, response, username, password);

            authenticated = true;
        } catch (ExpiredCredentialException ece) {
            logger.info("Credentials have expired or were reset by an administrator", ece);
            authenticated = false;
            response.addHeader(PASSWORD_RESET_REQUIRED_HEADER, "true");
        } catch (Exception e) {
            logger.info(e.getMessage(), e);
            authenticated = false;
        }

        // set password to true so that when we call super.login(), the authentication is treated as successful
        // perform session/cookie authentication stuff for seraph
        String fakePassword = authenticated ? CORRECT_PASSWORD : INCORRECT_PASSWORD;

        logger.debug("Updating user session for Seraph");

        authenticated = super.login(request, response, username, fakePassword, cookie);

        return authenticated;
    }

    @Override
    public boolean logout(HttpServletRequest request, HttpServletResponse response) throws AuthenticatorException {
        try {
            logger.debug("Logging off from Crowd");

            // Invalidate the user in Crowd.
            crowdHttpAuthenticator.logout(request, response);

            // Logout the user, removing Crowd specific attributes
            logger.debug("Invalidating user in Crowd-Seraph specific session variables");
            logoutUser(request);
        } catch (Exception e) {
            logger.info(e.getMessage(), e);
        }

        // Logout via Seraph now.
        logger.debug("Invalidating user in Seraph specific session variables");
        return super.logout(request, response);
    }

    /**
     * Checks to see if the request can be authenticated. This method checks (in order):
     * <ol>
     * <li>
     * Trusted Apps: it is possible that an earlier filter authenticated the request,
     * so check to see if this is the case.
     * </li>
     * <li>
     * Crowd Authenticator: if a valid Crowd session-cookie (token) exists,
     * the HttpAuthenticator will authenticate the request as "valid". This will not
     * place the user into the session. See getUser() to see exactly when the user
     * gets placed into session.
     * </li>
     * <li>
     * Seraph-Remember Me: sees if the request is authenticated via a remember me cookie.
     * If it is, then the user will be automatically logged into session and a Crowd SSO
     * token will be generated and put on the response.
     * </li>
     * <li>
     * Basic Authentication: determines if the request has Basic Auth username/password headers
     * and proceeds to authenticate the user with Crowd if they are present. The user will be
     * automatically logged into session and a Crowd SSO token will be generated and put on the response.
     * </li>
     * </ol>
     * If all checks fail authentication, the isAuthenticated method returns false, and the user is logged out.
     *
     * @param request  servlet request.
     * @param response servlet response.
     * @return true if request can be authenticated.
     * @deprecated since 2.9.0. Use {@link #checkAuthenticated(HttpServletRequest, HttpServletResponse)} instead.
     */
    @Deprecated
    protected boolean isAuthenticated(HttpServletRequest request, HttpServletResponse response) {
        return checkAuthenticated(request, response).isAuthenticated();
    }

    /**
     * Checks to see if the request can be authenticated. This method checks (in order):
     * <ol>
     * <li>
     * Trusted Apps: it is possible that an earlier filter authenticated the request,
     * so check to see if this is the case.
     * </li>
     * <li>
     * Crowd Authenticator: if a valid Crowd session-cookie (token) exists,
     * the HttpAuthenticator will authenticate the request as "valid". This will not
     * place the user into the session. See getUser() to see exactly when the user
     * gets placed into session.
     * </li>
     * <li>
     * Seraph-Remember Me: sees if the request is authenticated via a remember me cookie.
     * If it is, then the user will be automatically logged into session and a Crowd SSO
     * token will be generated and put on the response.
     * </li>
     * <li>
     * Basic Authentication: determines if the request has Basic Auth username/password headers
     * and proceeds to authenticate the user with Crowd if they are present. The user will be
     * automatically logged into session and a Crowd SSO token will be generated and put on the response.
     * </li>
     * </ol>
     * If all checks fail authentication, the isAuthenticated method returns
     * <tt>false</tt>, and the user will be logged out.
     *
     * @param request  servlet request.
     * @param response servlet response.
     * @return authentication state of the request
     * @since 2.8.3
     */
    protected AuthenticationState checkAuthenticated(HttpServletRequest request, HttpServletResponse response) {
        AuthenticationState authenticationState;

        authenticationState = isTrustedAppsRequest(request) ? authenticated() : unauthenticated();

        if (!authenticationState.isAuthenticated()) {
            try {
                // try Crowd's HttpAuthenticator
                authenticationState = crowdHttpAuthenticator.checkAuthenticated(request, response);
                if (authenticationState.isAuthenticated() && logger.isDebugEnabled()) {
                    logger.debug("User IS authenticated via the Crowd session-token");
                } else if (logger.isDebugEnabled()) {
                    logger.debug("User is NOT authenticated via the Crowd session-token");
                }
            } catch (Exception e) {
                logger.info("Error while attempting to check if user isAuthenticated with Crowd", e);
            }
        }

        // if Crowd's HttpAuthenticator failed try AutoLogin
        // i.e. try authenticating the user using credentials from the auto-login cookie
        if (!authenticationState.isAuthenticated()) {
            authenticationState = checkRememberMeLoginToCrowd(request, response);
            if (authenticationState.isAuthenticated() && logger.isDebugEnabled()) {
                logger.debug("Authenticated via remember-me cookie");
            } else if (logger.isDebugEnabled()) {
                logger.debug("Failed to authenticate via remember-me cookie");
            }
        }

        // Attempt Basic-Auth
        if (!authenticationState.isAuthenticated()) {
            if (RedirectUtils.isBasicAuthentication(request, getAuthType())) {
                // this will try logging in using basic authentication (this will send a 401 + auth header
                // if there is no auth header in the request)
                Principal basicAuthUser = getUserFromBasicAuthentication(request, response);
                if (basicAuthUser != null) {
                    authenticationState = authenticated(basicAuthUser);
                }
            }
        }

        if (!authenticationState.isAuthenticated()) {
            if (request.getSession(false) != null) {
                logger.debug("Request is not authenticated, logging out the user");
                try {
                    logoutUser(request);
                    if (response != null) {
                        super.logout(request, response);
                    }
                } catch (AuthenticatorException e) {
                    logger.error(e.getMessage(), e);
                }
            } else {
                logger.debug("Request is not authenticated and has no session.");
            }

            authenticationState = unauthenticated();
        }

        return authenticationState;
    }

    /**
     * Attempts to authenticate the request based on the auto-login cookie (if set).
     * This will only authenticate to Crowd via HttpAuthenticator. This will not set
     * any session variables and the like.
     *
     * @param request  servlet request.
     * @param response servlet response.
     * @return true if authentication via HttpAuthenticator using auto-login credentials successful.
     * @deprecated since 2.9.0. Use {@link #checkRememberMeLoginToCrowd(HttpServletRequest, HttpServletResponse)} instead.
     */
    @Deprecated
    protected boolean rememberMeLoginToCrowd(HttpServletRequest request, HttpServletResponse response) {
        return checkRememberMeLoginToCrowd(request, response).isAuthenticated();
    }

    /**
     * Attempts to authenticate the request based on the auto-login cookie (if set).
     * This will only authenticate to Crowd via HttpAuthenticator. This will not set
     * any session variables and the like.
     *
     * @param request  servlet request.
     * @param response servlet response.
     * @return true if authentication via HttpAuthenticator using auto-login credentials successful.
     * @since 2.8.3
     */
    protected AuthenticationState checkRememberMeLoginToCrowd(HttpServletRequest request, HttpServletResponse response) {
        // this method puts the user in session if they are auth'd
        Principal cookieUser = getUserFromCookie(request, response);

        if (cookieUser == null) {
            // cookie verification failed
            return unauthenticated();
        } else {
            logger.debug("User successfully authenticated via remember-me cookie verification");

            // well, the cookie verification says that the user is logged in so we'll trust it
            try {
                User user = crowdHttpAuthenticator.authenticateWithoutValidatingPassword(
                        request, response, cookieUser.getName());

                return authenticated(user);
            } catch (Exception e) {
                logger.debug("Could not register remember-me cookie authenticated user with Crowd SSO: " + cookieUser.getName() + ", reason: " + e.getMessage(), e);

                // maybe because crowd is down or because the user has been removed from crowd / marked as inactive, ie. shouldn't let him in
                removePrincipalFromSessionContext(request);

                return unauthenticated();
            }
        }
    }

    /**
     * This method will allow you to remove all session information about the user and force them to re-authenticate
     * If you wish to remove specific application attributes for the user, e.g.
     * <code>org.acegisecurity.context.SecurityContextHolder.clearContext();</code> from Bamboo
     *
     * @param request the current request
     */
    protected abstract void logoutUser(HttpServletRequest request);

    @Override
    public Principal getUser(final HttpServletRequest request, HttpServletResponse response) {
        ElevatedSecurityGuard securityGuard = getElevatedSecurityGuard();

        Principal user = null;

        if (isTrustedAppsRequest(request)) {
            return getUserFromSession(request);
        }

        // isAuthenticated performs Crowd cookie verification, remember-me cookie auth and basic auth,
        // see the isAuthenticated() javadoc / impl for more information
        AuthenticationState authenticationState = checkAuthenticated(request, response);
        if (authenticationState.isAuthenticated()) {
            final String cookieToken = crowdHttpAuthenticator.getToken(request);
            if (cookieToken == null) {
                // log the error, and allow the method to return a null user
                logger.error("Could not find cookieToken from authenticated request");
                return null;
            }
            final Object sessionToken = request.getSession().getAttribute(SESSION_TOKEN_KEY);
            if (cookieToken.equals(sessionToken)) {
                user = getUserFromSession(request);
            }

            // if user does not already exist in the session (or is different to the user logged in via SSO), we put the auth'd user in there
            if (user == null) {
                try {
                    // find our existing token
                    Optional<Principal> crowdUser = authenticationState.getAuthenticatedPrincipal();
                    if (!crowdUser.isPresent()) {
                        crowdUser = Optional.<Principal>fromNullable(crowdHttpAuthenticator.getUser(request));
                    }

                    Optional<String> updatedUsername = triggerUserAuthenticatedNotification(crowdUser);

                    user = updatedUsername.transform(this::getUser).orNull();
                } catch (Exception e) {
                    logger.info(e.getMessage(), e);
                }

                if (user != null) {
                    // JRA-20210 Ensure the user is allowed to log in
                    if (authoriseUserAndEstablishSession(request, response, user)) {
                        OK.stampRequestResponse(request, response);
                        securityGuard.onSuccessfulLoginAttempt(request, user.getName());
                        request.getSession().setAttribute(SESSION_TOKEN_KEY, cookieToken);
                    } else {
                        return null;
                    }
                }
            } else {
                // user was obtained from the session and the crowd cookie hasn't changed since, so we're good to stamp him in
                OK.stampRequestResponse(request, response);
            }
        }

        return user;
    }

    /**
     * Notify potential remote directories user was logged in, this might pull the user over
     * from the remote directory into the local one, and needs to be done before further checks
     * are performed
     */
    private Optional<String> triggerUserAuthenticatedNotification(Optional<Principal> crowdUser) {
        final Optional<String> originalName = crowdUser.transform(Principal::getName);

        if (crowdUser.isPresent() && !DISABLE_USER_AUTHENTICATED_NOTIFICATIONS) {
            final String crowdUserName = crowdUser.get().getName();

            try {
                return userAuthenticatedSentCache.get(crowdUserName, () -> {
                    try {
                        logger.debug("User session for {} established via SSO, notifying CrowdService", crowdUserName);
                        final User updatedUser = crowdServiceSupplier.get().userAuthenticated(crowdUserName);
                        return Optional.of(updatedUser.getName());
                    } catch (InactiveAccountException e) {
                        logger.warn("User {} is inactive during CrowdService.userAuthenticated", crowdUserName, e);
                        return Optional.absent();
                    } catch (com.atlassian.crowd.exception.runtime.UserNotFoundException e) {
                        logger.warn("User {} not found during CrowdService.userAuthenticated", crowdUserName, e);
                        return Optional.absent();
                    } catch (com.atlassian.crowd.exception.runtime.OperationFailedException e) {
                        logger.warn("Error executing CrowdService.userAuthenticated for user {}, falling back to local user", crowdUserName, e);
                        // might be a transient connection issue, fall back to the original user
                        return originalName;
                    }
                });
            } catch (ExecutionException | UncheckedExecutionException e) {
                logger.warn("Error executing userAuthenticated for user {}", crowdUserName, e);
            }
        }

        return originalName;
    }

    private boolean isTrustedAppsRequest(HttpServletRequest request) {
        if (BaseLoginFilter.LOGIN_SUCCESS.equals(request.getAttribute(BaseLoginFilter.OS_AUTHSTATUS_KEY))) {
            if (logger.isDebugEnabled()) {
                logger.debug("User IS authenticated via previous filter/trusted apps");
            }
            return true;
        }

        return false;
    }
}