package com.atlassian.crowd.integration.http;

import com.atlassian.crowd.exception.ApplicationAccessDeniedException;
import com.atlassian.crowd.exception.ApplicationPermissionException;
import com.atlassian.crowd.exception.CrowdException;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidTokenException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.integration.AuthenticationState;
import com.atlassian.crowd.integration.http.util.CrowdHttpTokenHelper;
import com.atlassian.crowd.model.authentication.CookieConfiguration;
import com.atlassian.crowd.model.authentication.Session;
import com.atlassian.crowd.model.authentication.UserAuthenticationContext;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.service.client.ClientProperties;
import com.atlassian.crowd.service.client.CrowdClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.security.Principal;
import java.util.Date;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;

/**
 * <p>An implementation of CrowdHttpAuthenticator using a {@link CrowdClient}
 * to talk to a Crowd server. All methods potentially result in calls to a
 * remote Crowd server.</p>
 * <p>If the {@link ClientProperties} provided has a session validation
 * interval configured then {@link #isAuthenticated(HttpServletRequest, HttpServletResponse)}
 * will only make remote calls when that interval expires.</p>
 */
public class CrowdHttpAuthenticatorImpl implements CrowdHttpAuthenticator {
    private static final Logger LOGGER = LoggerFactory.getLogger(CrowdHttpAuthenticator.class);
    private final CrowdClient client;
    private final ClientProperties clientProperties;
    private final CrowdHttpTokenHelper tokenHelper;
    private final TokenLockProvider tokenLockProvider;

    public CrowdHttpAuthenticatorImpl(CrowdClient client, ClientProperties clientProperties, CrowdHttpTokenHelper tokenHelper) {
        this(client, clientProperties, tokenHelper, new NoOpTokenLockProvider());
    }

    public CrowdHttpAuthenticatorImpl(final CrowdClient client, final ClientProperties clientProperties, final CrowdHttpTokenHelper tokenHelper, final TokenLockProvider tokenLockProvider) {
        this.client = client;
        this.clientProperties = clientProperties;
        this.tokenHelper = tokenHelper;
        this.tokenLockProvider = tokenLockProvider;
    }

    @Override
    public User getUser(final HttpServletRequest request)
            throws InvalidTokenException, ApplicationPermissionException, InvalidAuthenticationException, OperationFailedException {
        String ssoToken = tokenHelper.getCrowdToken(request, getCookieTokenKey());
        if (ssoToken != null) {
            return client.findUserFromSSOToken(ssoToken);
        } else {
            LOGGER.debug("Could not find user from token.");
            return null;
        }
    }

    @Override
    public User authenticate(final HttpServletRequest request, final HttpServletResponse response, final String username, final String password)
            throws InvalidTokenException, ApplicationAccessDeniedException, ExpiredCredentialException, InactiveAccountException, ApplicationPermissionException, InvalidAuthenticationException, OperationFailedException {
        UserAuthenticationContext userAuthenticationContext = tokenHelper.getUserAuthenticationContext(request, username, password, clientProperties);
        CookieConfiguration cookieConfig = client.getCookieConfiguration();
        String ssoToken = null;
        try {
            ssoToken = client.authenticateSSOUser(userAuthenticationContext);
            tokenHelper.setCrowdToken(request, response, ssoToken, clientProperties, cookieConfig);
        } finally {
            // clean up the session information if the authentication shouldn't be allowed
            if (ssoToken == null) {
                tokenHelper.removeCrowdToken(request, response, clientProperties, cookieConfig);
            }
        }

        return client.findUserFromSSOToken(ssoToken);
    }

    @Override
    public User authenticateWithoutValidatingPassword(final HttpServletRequest request, final HttpServletResponse response, final String username)
            throws InvalidTokenException, ApplicationAccessDeniedException, InactiveAccountException, ApplicationPermissionException, InvalidAuthenticationException, OperationFailedException {
        UserAuthenticationContext userAuthenticationContext = tokenHelper.getUserAuthenticationContext(request, username, null, clientProperties);
        CookieConfiguration cookieConfig = client.getCookieConfiguration();
        String ssoToken = null;
        try {
            ssoToken = client.authenticateSSOUserWithoutValidatingPassword(userAuthenticationContext);
            tokenHelper.setCrowdToken(request, response, ssoToken, clientProperties, cookieConfig);
        } finally {
            // clean up the session information if the authentication shouldn't be allowed
            if (ssoToken == null) {
                tokenHelper.removeCrowdToken(request, response, clientProperties, cookieConfig);
            }
        }

        return client.findUserFromSSOToken(ssoToken);
    }

    @Override
    @Deprecated
    public boolean isAuthenticated(HttpServletRequest request, HttpServletResponse response)
            throws OperationFailedException {
        return checkAuthenticated(request, response).isAuthenticated();
    }

    @Override
    public AuthenticationState checkAuthenticated(final HttpServletRequest request, final HttpServletResponse response) throws OperationFailedException {
        if (!shouldRevalidateSession(request)) {
            return AuthenticationState.authenticated();
        }

        // Check if we have a token, if it is not present, assume we are no longer authenticated
        final String token = getToken(request);
        if (token == null) {
            LOGGER.debug("Non authenticated request, unable to find a valid Crowd token.");
            return AuthenticationState.unauthenticated();
        }
        final Lock lockForToken = tokenLockProvider.getLock(token);
        lockForToken.lock();
        try {
            if (!shouldRevalidateSession(request)) { // we've entered the lock, other thread could have already revalidated the session, let's check this
                return AuthenticationState.authenticated();
            }
            final Session crowdSession = client.validateSSOAuthenticationAndGetSession(token, tokenHelper.getValidationFactorExtractor().getValidationFactors(request));
            final CookieConfiguration cookieConfig = client.getCookieConfiguration();
            tokenHelper.setCrowdToken(request, response, token, clientProperties, cookieConfig);
            final Principal principal = crowdSession.getUser();
            return AuthenticationState.authenticated(principal);
        } catch (ApplicationPermissionException | InvalidTokenException | InvalidAuthenticationException e) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(e.getMessage(), e);
            }
            // don't invalidate the client unless asked to do so (authentication status fails, or logoff).
            return AuthenticationState.unauthenticated();
        } finally {
            lockForToken.unlock();
        }
    }

    @Override
    public void logout(final HttpServletRequest request, final HttpServletResponse response)
            throws ApplicationPermissionException, InvalidAuthenticationException, OperationFailedException {
        CookieConfiguration cookieConfig = client.getCookieConfiguration();

        String ssoToken = tokenHelper.getCrowdToken(request, getCookieTokenKey(cookieConfig));
        if (ssoToken != null && !ssoToken.isEmpty()) {
            client.invalidateSSOToken(ssoToken);
        }

        tokenHelper.removeCrowdToken(request, response, clientProperties, cookieConfig);
    }


    @Override
    public String getToken(HttpServletRequest request) {
        return tokenHelper.getCrowdToken(request, getCookieTokenKey());
    }

    private String getCookieTokenKey(CookieConfiguration config) {
        return clientProperties.getCookieTokenKey(config.getName());
    }

    private String getCookieTokenKey() {
        String configuredKey = clientProperties.getCookieTokenKey(null);
        if (configuredKey != null) {
            return configuredKey;
        } else {
            try {
                return client.getCookieConfiguration().getName();
            } catch (CrowdException e) {
                LOGGER.info("Failed to get cookie configuration from remote Crowd", e);
                return clientProperties.getCookieTokenKey();
            } catch (ApplicationPermissionException e) {
                LOGGER.info("Failed to get cookie configuration from remote Crowd", e);
                return clientProperties.getCookieTokenKey();
            }
        }
    }

    private boolean shouldRevalidateSession(HttpServletRequest request) {
        final HttpSession session = request.getSession(false);
        if (session == null || clientProperties.getSessionValidationInterval() == 0) {
            return true;
        }
        final Date lastValidation;
        try {
            lastValidation = (Date) session.getAttribute(clientProperties.getSessionLastValidation());
        } catch (IllegalStateException e) {
            return true;
        }
        if (lastValidation != null) {
            // if the validation has previously been done, add the previous time plus the allowed interval
            long timeSpread = lastValidation.getTime() + TimeUnit.MINUTES.toMillis(clientProperties.getSessionValidationInterval());
            return timeSpread <= System.currentTimeMillis();
        }
        return true;
    }
}
