package com.atlassian.crowd.model.authentication;

import java.util.Objects;

/**
 * Represents the configuration of Crowd's cookie based SSO
 * @since v3.0
 */
public class CookieConfiguration {

    /**
     * This is the default SSO cookie name
     */
    public static String DEFAULT_COOKIE_TOKEN_KEY = "crowd.token_key";

    private final String domain;
    private final boolean secure;
    private final String name;

    public CookieConfiguration(final String domain, final boolean secure, final String name) {
        this.domain = domain;
        this.secure = secure;
        this.name = name;
    }

    /**
     * Returns the domain that the cookie should be set for.
     *
     * @return domain
     */
    public String getDomain() {
        return domain;
    }

    /**
     * Returns <tt>true</tt> if the cookie should only be included in a secure connection.
     *
     * @return <tt>true</tt> if the cookie should only be included in a secure connection
     */
    public boolean isSecure() {
        return secure;
    }

    /**
     * Returns the name of the SSO token cookie.
     *
     * @return name of the SSO token cookie
     */
    public String getName() {
        return name;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("CookieConfiguration{");
        sb.append("domain='").append(domain).append('\'');
        sb.append(", secure=").append(secure);
        sb.append(", name='").append(name).append('\'');
        sb.append('}');
        return sb.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        CookieConfiguration that = (CookieConfiguration) o;
        return secure == that.secure &&
                Objects.equals(domain, that.domain) &&
                Objects.equals(name, that.name);
    }

    @Override
    public int hashCode() {
        return Objects.hash(domain, secure, name);
    }
}
