package com.atlassian.crowd.search.query.membership;

import com.atlassian.crowd.embedded.api.Query;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.builder.ToStringBuilder;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import static com.atlassian.crowd.search.Entity.GROUP;
import static com.google.common.collect.Iterables.toArray;

public class MembershipQuery<T> implements Query<T> {
    private final EntityDescriptor entityToReturn;
    private final EntityDescriptor entityToMatch;
    /**
     * If true, we search for children of the entityToMatch, otherwise we are searching for its parents.
     */
    private final boolean findChildren;
    private final Set<String> entityNamesToMatch;
    private final int startIndex;
    private final int maxResults;
    private final Class<T> returnType;
    private final SearchRestriction searchRestriction;

    /**
     * @deprecated Use {@link #MembershipQuery(Class, boolean, EntityDescriptor, String, EntityDescriptor, int, int, SearchRestriction)}} instead. Since v2.9.
     */
    @Deprecated
    public MembershipQuery(final Class<T> returnType, final boolean findChildren, final EntityDescriptor entityToMatch, final String entityNameToMatch, final EntityDescriptor entityToReturn, final int startIndex, final int maxResults) {
        this(returnType, findChildren, entityToMatch, entityToReturn, startIndex, maxResults, QueryBuilder.NULL_RESTRICTION, entityNameToMatch);
    }

    public MembershipQuery(final Class<T> returnType, final boolean findChildren, final EntityDescriptor entityToMatch, final String entityNameToMatch, final EntityDescriptor entityToReturn, final int startIndex, final int maxResults, final SearchRestriction searchRestriction) {
        this(returnType, findChildren, entityToMatch, entityToReturn, startIndex, maxResults, searchRestriction, entityNameToMatch);
    }

    public MembershipQuery(final Class<T> returnType, final boolean findChildren, final EntityDescriptor entityToMatch, final EntityDescriptor entityToReturn, final int startIndex, final int maxResults, final SearchRestriction searchRestriction, final String... entityNamesToMatch) {
        this(returnType, findChildren, entityToMatch, entityToReturn, startIndex, maxResults, searchRestriction, validateAndCopyEntityNamesToMatch(entityNamesToMatch));
    }

    public MembershipQuery(final Class<T> returnType, final boolean findChildren, final EntityDescriptor entityToMatch, final EntityDescriptor entityToReturn, final int startIndex, final int maxResults, final SearchRestriction searchRestriction, final Collection<String> entityNamesToMatch) {
        Validate.notNull(entityToMatch, "entityToMatch argument cannot be null");
        Validate.notNull(entityNamesToMatch, "entityNamesToMatch argument cannot be null");
        Validate.notNull(entityToReturn, "entityToReturn argument cannot be null");
        Validate.isTrue(maxResults == EntityQuery.ALL_RESULTS || maxResults > 0, "maxResults must be greater than 0 (unless set to EntityQuery.ALL_RESULTS)");
        Validate.isTrue(startIndex >= 0, "startIndex cannot be less than zero");
        Validate.notNull(returnType, "returnType cannot be null");

        if (findChildren) {
            Validate.isTrue(entityToMatch.getEntityType() == GROUP, "Cannot find the children of type: " + entityToMatch);
        } else {
            Validate.isTrue(entityToReturn.getEntityType() == GROUP, "Cannot return parents of type: " + entityToMatch);
        }

        this.entityToReturn = entityToReturn;
        this.entityToMatch = entityToMatch;
        this.findChildren = findChildren;
        this.entityNamesToMatch = validateAndCopyEntityNamesToMatch(entityNamesToMatch);
        this.startIndex = startIndex;
        this.maxResults = maxResults;
        this.returnType = returnType;
        this.searchRestriction = searchRestriction;
    }

    private static Set<String> validateAndCopyEntityNamesToMatch(Collection<String> entityNamesToMatch) {
        Validate.noNullElements(entityNamesToMatch, "entityNamesToMatch argument cannot contain any null elements");
        return ImmutableSet.copyOf(entityNamesToMatch);
    }

    private static Set<String> validateAndCopyEntityNamesToMatch(String... entityNamesToMatch) {
        Validate.noNullElements(entityNamesToMatch, "entityNamesToMatch argument cannot contain any null elements");
        return ImmutableSet.copyOf(entityNamesToMatch);
    }

    public MembershipQuery(final MembershipQuery<T> query, final int startIndex, final int maxResults) {
        this(query.getReturnType(), query.isFindChildren(), query.getEntityToMatch(), query.getEntityToReturn(), startIndex, maxResults, query.getSearchRestriction(), toArray(query.getEntityNamesToMatch(), String.class));
    }

    public MembershipQuery(final MembershipQuery<?> query, final Class<T> returnType) {
        this(returnType,
                query.isFindChildren(),
                query.getEntityToMatch(),
                query.getEntityToReturn(),
                query.getStartIndex(),
                query.getMaxResults(),
                query.getSearchRestriction(),
                toArray(query.getEntityNamesToMatch(), String.class)
        );
    }

    public EntityDescriptor getEntityToReturn() {
        return entityToReturn;
    }

    public EntityDescriptor getEntityToMatch() {
        return entityToMatch;
    }

    public boolean isFindChildren() {
        return findChildren;
    }

    public Set<String> getEntityNamesToMatch() {
        return entityNamesToMatch;
    }

    /**
     * This will return the entity name to match if {@link #getEntityNamesToMatch()} contains a single value or <code>null</code>.
     *
     * @return the entity name to match if {@link #getEntityNamesToMatch()} contains a single value or <code>null</code>.
     * @throws IllegalArgumentException if {@link #getEntityNamesToMatch()} has a size greater than one.
     * @deprecated Use {@link #getEntityNamesToMatch()} instead. Since v2.9
     */
    @Deprecated
    @Nullable
    public String getEntityNameToMatch() {
        return Iterables.getOnlyElement(entityNamesToMatch, null);
    }

    public int getStartIndex() {
        return startIndex;
    }

    public int getMaxResults() {
        return maxResults;
    }

    public Class<T> getReturnType() {
        return returnType;
    }

    public SearchRestriction getSearchRestriction() {
        return searchRestriction;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (!(o instanceof MembershipQuery)) return false;

        MembershipQuery that = (MembershipQuery) o;

        if (findChildren != that.findChildren) return false;
        if (maxResults != that.maxResults) return false;
        if (startIndex != that.startIndex) return false;
        if (!Objects.equals(this.entityNamesToMatch, that.entityNamesToMatch))
            return false;
        if (entityToMatch != null ? !entityToMatch.equals(that.entityToMatch) : that.entityToMatch != null)
            return false;
        if (entityToReturn != null ? !entityToReturn.equals(that.entityToReturn) : that.entityToReturn != null)
            return false;
        if (returnType != that.returnType) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = entityToReturn != null ? entityToReturn.hashCode() : 0;
        result = 31 * result + (entityToMatch != null ? entityToMatch.hashCode() : 0);
        result = 31 * result + (findChildren ? 1 : 0);
        result = 31 * result + (entityNamesToMatch != null ? Objects.hashCode(entityNamesToMatch) : 0);
        result = 31 * result + startIndex;
        result = 31 * result + maxResults;
        result = 31 * result + (returnType != null ? returnType.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this).
                append("entityToReturn", entityToReturn).
                append("entityToMatch", entityToMatch).
                append("findChildren", findChildren).
                append("entityNamesToMatch", Iterables.toString(entityNamesToMatch)).
                append("startIndex", startIndex).
                append("maxResults", maxResults).
                append("returnType", returnType.getSimpleName()).
                toString();
    }

    public MembershipQuery<T> withEntityNames(Collection<String> entityNamesToMatch) {
        return new MembershipQuery<>(returnType, findChildren, entityToMatch, entityToReturn, startIndex, maxResults,
                searchRestriction, entityNamesToMatch);
    }

    public MembershipQuery<T> withEntityNames(String... entityNameToMatch) {
        return withEntityNames(validateAndCopyEntityNamesToMatch(entityNameToMatch));
    }

    /**
     * Splits query with multiple {@link #getEntityNamesToMatch()} into separate queries with single entity name to
     * match.
     * Returned queries will have: {@link #startIndex} equal to 0 and {@link #maxResults} equal to original
     * {@code startIndex + maxResults}. This is required to correctly merge and produce results of original query.
     */
    public List<MembershipQuery<T>> splitEntityNamesToMatch() {
        return splitEntityNamesToMatch(1);
    }

    /**
     * Splits query with multiple {@link #getEntityNamesToMatch()} into separate queries with provided number
     * of entity names to match.
     * Returned queries will have: {@link #startIndex} equal to 0 and {@link #maxResults} equal to original
     * {@code startIndex + maxResults}. This is required to correctly merge and produce results of original query.
     */
    public List<MembershipQuery<T>> splitEntityNamesToMatch(int batchSize) {
        MembershipQuery<T> base = baseSplitQuery();
        return Lists.partition(new ArrayList<>(entityNamesToMatch), batchSize).stream()
                .map(base::withEntityNames)
                .collect(Collectors.toList());
    }

    public MembershipQuery<T> withAllResults() {
        return withStartIndexAndMaxResult(0, EntityQuery.ALL_RESULTS);
    }

    public MembershipQuery<T> withStartIndex(int startIndex) {
        return withStartIndexAndMaxResult(startIndex, maxResults);
    }

    public MembershipQuery<T> withMaxResults(int maxResults) {
        return withStartIndexAndMaxResult(startIndex, maxResults);
    }

    public <Q> MembershipQuery<Q> withReturnType(Class<Q> returnType) {
        return new MembershipQuery<>(returnType, findChildren, entityToMatch, entityToReturn, startIndex, maxResults,
                searchRestriction, entityNamesToMatch);
    }

    public MembershipQuery<T> withEntityToReturn(EntityDescriptor entityToReturn) {
        return new MembershipQuery<>(returnType, findChildren, entityToMatch, entityToReturn, startIndex, maxResults,
                searchRestriction, entityNamesToMatch);
    }

    /**
     * Returned queries will have": {@link #startIndex} equal to 0 and {@link #maxResults} equal to original
     * {@code startIndex + maxResults}. This is required to correctly merge and produce results of original query.
     */
    public MembershipQuery<T> baseSplitQuery() {
        return withStartIndexAndMaxResult(0, EntityQuery.addToMaxResults(this.maxResults, startIndex));
    }

    public MembershipQuery<T> addToMaxResults(int add) {
        return withMaxResults(EntityQuery.addToMaxResults(this.maxResults, add));
    }

    public MembershipQuery<T> withStartIndexAndMaxResult(int startIndex, int maxResults) {
        return new MembershipQuery<>(returnType, findChildren, entityToMatch, entityToReturn, startIndex, maxResults,
                searchRestriction, entityNamesToMatch);    }

    public MembershipQuery<T> withSearchRestriction(SearchRestriction searchRestriction) {
        return new MembershipQuery<>(returnType, findChildren, entityToMatch, entityToReturn, startIndex, maxResults,
                searchRestriction, entityNamesToMatch);
    }

    public boolean isWithAllResults() {
        return getStartIndex() == 0 && getMaxResults() == EntityQuery.ALL_RESULTS;
    }
}
