package com.atlassian.crowd.model.directory;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectoryType;
import com.atlassian.crowd.embedded.api.OperationType;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.model.InternalEntity;
import com.atlassian.crowd.model.InternalEntityTemplate;
import com.atlassian.crowd.util.InternalEntityUtils;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.builder.ToStringBuilder;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;

/**
 * Implementation of Directory (designed for use with Hibernate).
 * @deprecated Use {@link ImmutableDirectory} instead. Since 3.6.0
 */
@Deprecated
public class DirectoryImpl extends InternalEntity implements Directory {
    public static final String ATTRIBUTE_KEY_USER_ENCRYPTION_METHOD = "user_encryption_method";
    public static final String ATTRIBUTE_KEY_USE_NESTED_GROUPS = "useNestedGroups";

    /**
     * Key to decide whether we support local user status for a given LDAP directory. If the value is
     * true, users can be enabled and disabled independently in Crowd and the LDAP server. If the value is
     * false, user status is synchronised between Crowd and the LDAP server (for LDAP directories that support
     * synchronisation of user status, like Active Directory). This option is only relevant for cached
     * directories.
     */
    public static final String ATTRIBUTE_KEY_LOCAL_USER_STATUS = "localUserStatusEnabled";
    public static final String ATTRIBUTE_KEY_AUTO_ADD_GROUPS = "autoAddGroups";
    public static final String ATTRIBUTE_KEY_USE_PRIMARY_GROUP = "ldap.activedirectory.use_primary_groups";

    /**
     * Key to decide whether synchronisable user attributes should be synchronised between the remote directory
     * and the local cache. This option is only relevant for cached directories. If absent, it is assumed to be
     * false.
     */
    public static final String ATTRIBUTE_KEY_USER_ATTRIBUTES_SYNC_ENABLED = "userAttributesSyncEnabled";

    /**
     * Key to decide whether synchronisable group attributes should be synchronised between the remote directory
     * and the local cache. This option is only relevant for cached directories. If absent, it is assumed to be
     * false.
     */
    public static final String ATTRIBUTE_KEY_GROUP_ATTRIBUTES_SYNC_ENABLED = "groupAttributesSyncEnabled";


    /**
     * Key of attribute which holds the timestamp of the last configuration change.
     * should be full
     */
    public static final String LAST_CONFIGURATION_CHANGE = "configuration.change.timestamp";

    public static final char AUTO_ADD_GROUPS_SEPARATOR = '|';
    public static final Set<String> PASSWORD_ATTRIBUTES = ImmutableSet.of(
            "ldap.password", "application.password", "AZURE_AD_WEBAPP_CLIENT_SECRET",
            "crowd.server.http.proxy.password");
    public static final String SANITISED_PASSWORD = PasswordCredential.SANITISED_PASSWORD;

    private String lowerName;
    private String description;
    private DirectoryType type;
    private String implementationClass;
    private String lowerImplementationClass;

    private Set<OperationType> allowedOperations = new HashSet<OperationType>();
    private Map<String, String> attributes = new HashMap<String, String>();

    public DirectoryImpl() {
    }

    /**
     * Used for importing via XML migration.
     *
     * @param template directory template.
     */
    public DirectoryImpl(InternalEntityTemplate template) {
        super(template);
    }

    public DirectoryImpl(String name, DirectoryType type, String implementationClass) {
        setName(name);
        setType(type);
        setImplementationClass(implementationClass);
        setActive(true);
    }

    public DirectoryImpl(Directory directory) {
        super(new InternalEntityTemplate(directory.getId(), directory.getName(), directory.isActive(), directory.getCreatedDate(), directory.getUpdatedDate()));
        setName(directory.getName());
        setType(directory.getType());
        setImplementationClass(directory.getImplementationClass());
        setActive(directory.isActive());
        setDescription(directory.getDescription());
        updateAllowedOperationsFrom(directory.getAllowedOperations());
        updateAttributesFrom(directory.getAttributes());
    }

    public void updateDetailsFrom(Directory directory) {
        setName(directory.getName());
        setType(directory.getType());
        setImplementationClass(directory.getImplementationClass());
        setActive(directory.isActive());
        setDescription(directory.getDescription());
        updateAllowedOperationsFrom(directory.getAllowedOperations());
        updateAttributesFrom(directory.getAttributes());
    }

    public void updateAttributesFrom(Map<String, String> attributes) {
        // Avoid discarding original collection, so that Hibernate won't perform one shot delete.
        this.attributes.entrySet().retainAll(attributes.entrySet());
        this.attributes.putAll(Maps.filterValues(attributes, value -> !Strings.isNullOrEmpty(value)));
    }

    public String getEncryptionType() {
        String encryptionType;

        // do we have an InternalDirectory?
        if ("com.atlassian.crowd.directory.InternalDirectory".equals(getImplementationClass())) {
            encryptionType = getValue(ATTRIBUTE_KEY_USER_ENCRYPTION_METHOD);
        } else {
            encryptionType = getValue("ldap.user.encryption");
        }

        return encryptionType;
    }

    public Map<String, String> getAttributes() {
        return attributes;
    }

    /**
     * Sets the attributes of the directory. <code>attributes</code> must be a mutable <tt>Map</tt>.
     *
     * @param attributes new attributes
     */
    public void setAttributes(final Map<String, String> attributes) {
        this.attributes = attributes;
    }

    public Set<OperationType> getAllowedOperations() {
        return allowedOperations;
    }

    public void addAllowedOperation(OperationType operationType) {
        getAllowedOperations().add(operationType);
    }

    public void setAllowedOperations(final Set<OperationType> allowedOperations) {
        this.allowedOperations = allowedOperations;
    }

    public void updateAllowedOperationsFrom(final Set<OperationType> allowedOperations) {
        this.allowedOperations.retainAll(allowedOperations);
        this.allowedOperations.addAll(allowedOperations);
    }

    // this private method is actually used by Hibernate/Spring
    @SuppressWarnings({"UnusedDeclaration"})
    private void setLowerImplementationClass(final String lowerImplementationClass) {
        this.lowerImplementationClass = lowerImplementationClass;
    }

    public String getLowerImplementationClass() {
        return lowerImplementationClass;
    }

    public String getLowerName() {
        return lowerName;
    }

    // this private method is actually used by Hibernate/Spring
    @SuppressWarnings({"UnusedDeclaration"})
    private void setLowerName(final String lowerName) {
        this.lowerName = lowerName;
    }

    public String getDescription() {
        return description;
    }

    public DirectoryType getType() {
        return type;
    }

    public String getImplementationClass() {
        return implementationClass;
    }

    public void setDescription(final String description) {
        this.description = description;
    }

    public void setType(final DirectoryType type) {
        Validate.notNull(type);
        this.type = type;
    }

    public void setImplementationClass(final String implementationClass) {
        Validate.notNull(implementationClass);
        this.implementationClass = implementationClass;
        this.lowerImplementationClass = implementationClass.toLowerCase(Locale.ENGLISH);
    }

    public void setName(final String name) {
        Validate.notNull(name);
        InternalEntityUtils.validateLength(name);
        this.name = name;
        this.lowerName = toLowerCase(name);
    }

    public void setActive(final boolean active) {
        this.active = active;
    }

    /**
     * @param name attribute name.
     * @return a collection of the only attribtue value or <code>null</code>
     * if the directory does not have the attribute.
     */
    public Set<String> getValues(final String name) {
        String value = getValue(name);
        if (value != null) {
            return Collections.singleton(value);
        } else {
            return null;
        }
    }

    public String getValue(final String name) {
        return attributes.get(name);
    }

    public Set<String> getKeys() {
        return attributes.keySet();
    }

    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    public void setAttribute(final String name, final String value) {
        attributes.put(name, value);
    }

    public void removeAttribute(final String name) {
        attributes.remove(name);
    }

    public void validate() {
        Validate.notEmpty(name, "name cannot be null");
        Validate.isTrue(toLowerCase(name).equals(lowerName), "lowerName must be the lower-case representation of name");
        Validate.notNull(type, "type cannot be null");
        Validate.notNull(implementationClass, "implementationClass cannot be null");
        Validate.notNull(createdDate, "createdDate cannot be null");
        Validate.notNull(updatedDate, "updatedDate cannot be null");
        Validate.notNull(allowedOperations, "allowedOperations cannot be null");
        Validate.notNull(attributes, "attributes cannot be null");
    }

    @SuppressWarnings({"RedundantIfStatement"})
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof DirectoryImpl)) {
            return false;
        }

        DirectoryImpl directory = (DirectoryImpl) o;

        if (getImplementationClass() != null ? !getImplementationClass().equals(directory.getImplementationClass()) : directory.getImplementationClass() != null) {
            return false;
        }
        if (getLowerName() != null ? !getLowerName().equals(directory.getLowerName()) : directory.getLowerName() != null) {
            return false;
        }
        if (getType() != directory.getType()) {
            return false;
        }

        return true;
    }

    public int hashCode() {
        int result = getLowerName() != null ? getLowerName().hashCode() : 0;
        result = 31 * result + (getType() != null ? getType().hashCode() : 0);
        result = 31 * result + (getImplementationClass() != null ? getImplementationClass().hashCode() : 0);
        return result;
    }

    public final String toString() {
        Map<String, String> attrs = getAttributes();

        attrs = new HashMap<String, String>(attrs);
        for (String a : PASSWORD_ATTRIBUTES) {
            if (attrs.containsKey(a)) {
                attrs.put(a, SANITISED_PASSWORD);
            }
        }

        return new ToStringBuilder(this).
                append("lowerName", getLowerName()).
                append("description", getDescription()).
                append("type", getType()).
                append("implementationClass", getImplementationClass()).
                append("allowedOperations", getAllowedOperations()).
                append("attributes", attrs).
                toString();
    }
}
