package com.atlassian.crowd.directory;

import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.collect.ListMultimap;

import javax.annotation.Nonnull;
import java.util.List;

/**
 * Interface supporting searches with multiple values to match.
 */
public interface MultiValuesQueriesSupport {
    /**
     * Searches for membership information.
     * Similar to {@link RemoteDirectory#searchGroupRelationships(MembershipQuery)}, but guarantees that multiple
     * {@link MembershipQuery#getEntityNamesToMatch()} are supported.
     *
     * @param query query for memberships.
     * @return a List of Users or Groups or Strings depending on the query criteria. An empty List if there are no
     * results. Results are ordered by entity name, case-insensitive.
     * @throws OperationFailedException underlying directory implementation failed to execute the operation.
     * @throws IllegalArgumentException if the query is not a valid membership query
     */
    @Nonnull
    <T> List<T> searchGroupRelationships(MembershipQuery<T> query) throws OperationFailedException;

    /**
     * Searches for {@link Group groups} that match the supplied query criteria.
     * This method is similar to {@link #searchGroupRelationships(MembershipQuery)}, but it additionally groups results
     * by elements of {@link MembershipQuery#getEntityNamesToMatch}.
     * <p>
     *
     * @param query EntityQuery for Entity.GROUP.
     * @return <code>ListMultimap&lt;String, Group&gt;</code> or <code>ListMultimap&lt;String, String&gt;</code> of groups/groupnames
     * matching the search criteria, grouped by {@link MembershipQuery#getEntityNamesToMatch}.
     * Results are ordered by entity name, case-insensitive.
     * An empty <code>{@link ListMultimap}</code> will be returned if no groups matching the criteria are found.
     * @throws OperationFailedException if the underlying directory implementation failed to execute the operation
     * @throws IllegalArgumentException if the query is not a valid group query
     */
    <T> ListMultimap<String, T> searchGroupRelationshipsGroupedByName(final MembershipQuery<T> query);
}
