package com.atlassian.crowd.model.group;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;

import javax.annotation.Nonnull;
import java.util.Objects;

public class ImmutableGroup implements Group {
    private final long directoryId;
    private final String name;
    private final GroupType type;
    private final boolean active;
    private final String description;
    private final String externalId;

    private ImmutableGroup(final Builder builder) {
        this.directoryId = builder.directoryId;
        this.name = builder.name;
        this.type = builder.type;
        this.active = builder.active;
        this.description = builder.description;
        this.externalId = builder.externalId;
    }

    /**
     * @return an {@link ImmutableGroup} with the same properties as the given group.
     * Will avoid creating a copy if possible.
     */
    public static ImmutableGroup from(final Group group) {
        if (group instanceof ImmutableGroup) {
            return (ImmutableGroup) group;
        }

        return ImmutableGroup.builder(group).build();
    }

    public static Builder builder(long directoryId, String name) {
        return new Builder(directoryId, name);
    }

    public static Builder builder(String name) {
        return new Builder(name);
    }

    public static Builder builder(Group group) {
        return new Builder(group);
    }

    @Override
    public int compareTo(@Nonnull Group other) {
        return GroupComparator.compareTo(this, other);
    }

    @Override
    public long getDirectoryId() {
        return directoryId;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public GroupType getType() {
        return type;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String getExternalId() {
        return externalId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImmutableGroup that = (ImmutableGroup) o;
        return directoryId == that.directoryId &&
                active == that.active &&
                Objects.equals(name, that.name) &&
                type == that.type &&
                Objects.equals(description, that.description);
    }

    @Override
    public int hashCode() {
        return Objects.hash(directoryId, name, type, active, description);
    }

    public static class Builder {
        private long directoryId;
        private String name;
        private GroupType type = GroupType.GROUP;
        private boolean active = true;
        private String description = "";
        private String externalId;

        public Builder(Group group) {
            Preconditions.checkNotNull(group, "group");
            setDirectoryId(group.getDirectoryId());
            setName(group.getName());
            setDescription(group.getDescription());
            setType(group.getType());
            setActive(group.isActive());
            setDescription(group.getDescription());
            setExternalId(group.getExternalId());
        }

        public Builder(String name) {
            this(-1L, name);
        }

        public Builder(long directoryId, String name) {
            this.directoryId = directoryId;
            this.name = name;
        }

        public Builder setDirectoryId(long directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        public Builder setType(GroupType type) {
            this.type = type;
            return this;
        }

        public Builder setActive(boolean active) {
            this.active = active;
            return this;
        }

        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        public Builder setExternalId(String externalId) {
            this.externalId = externalId;
            return this;
        }

        public ImmutableGroup build() {
            return new ImmutableGroup(this);
        }
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("directoryId", directoryId)
                .add("name", name)
                .add("type", type)
                .add("active", active)
                .add("description", description)
                .add("externalId", externalId)
                .toString();
    }
}
