package com.atlassian.crowd.model.application;

import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.model.webhook.ImmutableWebhook;
import com.atlassian.crowd.model.webhook.Webhook;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Immutable Application
 *
 * @since 2.2
 */
public final class ImmutableApplication implements Application {
    private final Long id;
    private final String name;
    private final ApplicationType type;
    private final String description;
    private final PasswordCredential passwordCredential;
    private final boolean permanent;
    private final boolean active;
    private final Map<String, String> attributes;
    private final List<ApplicationDirectoryMapping> directoryMappings;
    private final Set<RemoteAddress> remoteAddresses;
    private final Set<ImmutableWebhook> webhooks;
    private final boolean lowercaseOutput;
    private final boolean aliasingEnabled;
    private final boolean filteringUsersWithAccessEnabled;
    private final boolean filteringGroupsWithAccessEnabled;
    private final boolean membershipAggregationEnabled;
    private final boolean cachedDirectoriesAuthenticationOrderOptimisationEnabled;
    private final boolean authenticationWithoutPasswordEnabled;
    private final Date createdDate;
    private final Date updatedDate;

    /**
     * @deprecated Use {@link com.atlassian.crowd.model.application.ImmutableApplication.Builder} instead.
     */
    @Deprecated
    public ImmutableApplication(final Long id, final String name, final ApplicationType type, final String description,
                                final PasswordCredential passwordCredential, final boolean permanent,
                                final boolean active, final Map<String, String> attributes,
                                final List<DirectoryMapping> directoryMappings,
                                final Set<RemoteAddress> remoteAddresses,
                                final Set<Webhook> webhooks,
                                final boolean lowercaseOutput,
                                final boolean aliasingEnabled,
                                final Date createdDate, final Date updatedDate) {
        this(id, name, type, description, passwordCredential, permanent, active, attributes,
                Lists.transform(directoryMappings, ImmutableApplicationDirectoryMapping::from),
                remoteAddresses, webhooks, lowercaseOutput, aliasingEnabled, false, false, false, false, false,
                createdDate, updatedDate);
    }

    private ImmutableApplication(final Long id, final String name, final ApplicationType type, final String description,
                                 final PasswordCredential passwordCredential, final boolean permanent,
                                 final boolean active, final Map<String, String> attributes,
                                 final List<ApplicationDirectoryMapping> directoryMappings,
                                 final Set<RemoteAddress> remoteAddresses,
                                 final Set<Webhook> webhooks,
                                 final boolean lowercaseOutput,
                                 final boolean aliasingEnabled,
                                 final boolean membershipAggregationEnabled,
                                 final boolean cachedDirectoriesAuthenticationOrderOptimisationEnabled,
                                 final boolean authenticationWithoutPasswordEnabled,
                                 final boolean filteringUsersWithAccessEnabled,
                                 final boolean filteringGroupsWithAccessEnabled,
                                 final Date createdDate, final Date updatedDate) {
        this.id = id;
        this.name = name;
        this.type = type;
        this.description = description;
        this.passwordCredential = passwordCredential;
        this.permanent = permanent;
        this.active = active;
        this.attributes = (attributes == null ? Collections.emptyMap() : ImmutableMap.copyOf(attributes));
        this.directoryMappings = directoryMappings == null ? Collections.emptyList() : ImmutableList.copyOf(directoryMappings);
        this.remoteAddresses = (remoteAddresses == null ? Collections.emptySet() : ImmutableSet.copyOf(remoteAddresses));
        this.webhooks = (webhooks == null ? Collections.emptySet() :
                ImmutableSet.copyOf(webhooks.stream()
                        .map(ImmutableWebhook::builder)
                        .map(b -> b.setApplication(this).build())
                        .collect(Collectors.toSet()))
        );
        this.lowercaseOutput = lowercaseOutput;
        this.aliasingEnabled = aliasingEnabled;
        this.membershipAggregationEnabled = membershipAggregationEnabled;
        this.cachedDirectoriesAuthenticationOrderOptimisationEnabled = cachedDirectoriesAuthenticationOrderOptimisationEnabled;
        this.authenticationWithoutPasswordEnabled = authenticationWithoutPasswordEnabled;
        this.filteringUsersWithAccessEnabled = filteringUsersWithAccessEnabled;
        this.filteringGroupsWithAccessEnabled = filteringGroupsWithAccessEnabled;
        this.createdDate = (createdDate == null ? null : new Date(createdDate.getTime()));
        this.updatedDate = (updatedDate == null ? null : new Date(updatedDate.getTime()));
    }

    @Override
    public Long getId() {
        return id;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public ApplicationType getType() {
        return type;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public PasswordCredential getCredential() {
        return passwordCredential == null ? null : new PasswordCredential(passwordCredential);
    }

    @Override
    public boolean isPermanent() {
        return permanent;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public Map<String, String> getAttributes() {
        return attributes;
    }

    @Override
    public List<DirectoryMapping> getDirectoryMappings() {
        return getApplicationDirectoryMappings().stream()
                .map(applicationDirectoryMapping -> DirectoryMapping.fromApplicationDirectoryMapping(this, applicationDirectoryMapping))
                .collect(Collectors.toList());
    }

    @Override
    @Nonnull
    public List<ApplicationDirectoryMapping> getApplicationDirectoryMappings() {
        return directoryMappings;
    }

    @Override
    public DirectoryMapping getDirectoryMapping(final long directoryId) {
        final ApplicationDirectoryMapping mapping = getApplicationDirectoryMapping(directoryId);
        return mapping == null ? null : DirectoryMapping.fromApplicationDirectoryMapping(this, mapping);
    }

    @Override
    public ApplicationDirectoryMapping getApplicationDirectoryMapping(final long directoryId) {
        final Long dirId = directoryId;
        return directoryMappings.stream()
                .filter(mapping -> dirId.equals(mapping.getDirectory().getId()))
                .findFirst().orElse(null);
    }

    @Override
    public Set<RemoteAddress> getRemoteAddresses() {
        return remoteAddresses;
    }

    @Override
    public boolean hasRemoteAddress(final String remoteAddress) {
        return remoteAddresses.contains(new RemoteAddress(remoteAddress));
    }

    @Override
    public Set<Webhook> getWebhooks() {
        return Collections.unmodifiableSet(webhooks);
    }

    @Override
    public boolean isLowerCaseOutput() {
        return lowercaseOutput;
    }

    @Override
    public boolean isAliasingEnabled() {
        return aliasingEnabled;
    }

    @Override
    public boolean isFilteringUsersWithAccessEnabled() {
        return filteringUsersWithAccessEnabled;
    }

    @Override
    public boolean isFilteringGroupsWithAccessEnabled() {
        return filteringGroupsWithAccessEnabled;
    }

    @Override
    public boolean isMembershipAggregationEnabled() {
        return membershipAggregationEnabled;
    }

    @Override
    public boolean isCachedDirectoriesAuthenticationOrderOptimisationEnabled() {
        return cachedDirectoriesAuthenticationOrderOptimisationEnabled;
    }

    @Override
    public boolean isAuthenticationWithoutPasswordEnabled() {
        return authenticationWithoutPasswordEnabled;
    }

    @Override
    public Date getCreatedDate() {
        return (createdDate == null ? null : new Date(createdDate.getTime()));
    }

    @Override
    public Date getUpdatedDate() {
        return (updatedDate == null ? null : new Date(updatedDate.getTime()));
    }

    @Override
    public Set<String> getValues(final String key) {
        final String value = attributes.get(key);
        return (value == null ? Collections.emptySet() : Collections.singleton(value));
    }

    @Override
    public String getValue(final String key) {
        return attributes.get(key);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    /**
     * Constructs a new builder for an <tt>ImmutableApplication</tt>.
     *
     * @param name name of the application
     * @param type type of the application
     * @return builder with the name and type initialised
     */
    public static Builder builder(final String name, final ApplicationType type) {
        return new Builder(name, type);
    }

    /**
     * Constructs a new builder for an <tt>ImmutableApplication</tt> with the fields initialised to
     * <code>application</code>.
     *
     * @param application application to duplicate
     * @return builder with the fields initialised to <code>application</code>
     */
    public static Builder builder(final Application application) {
        return new Builder(application);
    }

    /**
     * @return an {@link ImmutableApplication} with the same properties as the given application.
     * Will avoid creating a copy if possible.
     */
    public static ImmutableApplication from(final Application application) {
        if (application instanceof ImmutableApplication) {
            return (ImmutableApplication) application;
        } else {
            return ImmutableApplication.builder(application).build();
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImmutableApplication that = (ImmutableApplication) o;
        return permanent == that.permanent &&
                active == that.active &&
                lowercaseOutput == that.lowercaseOutput &&
                aliasingEnabled == that.aliasingEnabled &&
                authenticationWithoutPasswordEnabled == that.authenticationWithoutPasswordEnabled &&
                filteringUsersWithAccessEnabled == that.filteringUsersWithAccessEnabled &&
                filteringGroupsWithAccessEnabled == that.filteringGroupsWithAccessEnabled &&
                membershipAggregationEnabled == that.membershipAggregationEnabled &&
                cachedDirectoriesAuthenticationOrderOptimisationEnabled == that.cachedDirectoriesAuthenticationOrderOptimisationEnabled &&
                Objects.equals(id, that.id) &&
                Objects.equals(name, that.name) &&
                type == that.type &&
                Objects.equals(description, that.description) &&
                Objects.equals(passwordCredential, that.passwordCredential) &&
                Objects.equals(attributes, that.attributes) &&
                Objects.equals(directoryMappings, that.directoryMappings) &&
                Objects.equals(remoteAddresses, that.remoteAddresses) &&
                Objects.equals(webhooks, that.webhooks) &&
                Objects.equals(createdDate, that.createdDate) &&
                Objects.equals(updatedDate, that.updatedDate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, name, type, description, passwordCredential, permanent,
                active, attributes, directoryMappings, remoteAddresses, webhooks,
                lowercaseOutput, aliasingEnabled, filteringUsersWithAccessEnabled, authenticationWithoutPasswordEnabled,
                filteringGroupsWithAccessEnabled, membershipAggregationEnabled,
                cachedDirectoriesAuthenticationOrderOptimisationEnabled, createdDate, updatedDate);
    }

    public static class Builder {
        private Long id;
        private String name;
        private ApplicationType type;
        private String description;
        private PasswordCredential passwordCredential;
        private boolean permanent;
        private boolean active;
        private Map<String, String> attributes;
        private List<ApplicationDirectoryMapping> directoryMappings;
        private Set<RemoteAddress> remoteAddresses;
        private Set<Webhook> webhooks;
        private boolean lowercaseOutput;
        private boolean aliasingEnabled;
        private boolean membershipAggregationEnabled;
        private boolean cachedDirectoriesAuthenticationOrderOptimisationEnabled;
        private boolean authenticationWithoutPasswordEnabled;
        private boolean filteringUsersWithAccessEnabled;
        private boolean filteringGroupsWithAccessEnabled;
        private Date createdDate;
        private Date updatedDate;

        public Builder(final String name, final ApplicationType type) {
            this.name = name;
            this.type = type;
        }

        public Builder(final Application application) {
            Preconditions.checkNotNull(application, "application");
            this.id = application.getId();
            this.name = application.getName();
            this.type = application.getType();
            this.description = application.getDescription();
            this.passwordCredential = application.getCredential();
            this.permanent = application.isPermanent();
            this.active = application.isActive();
            this.attributes = application.getAttributes();
            setDirectoryMappings(application.getDirectoryMappings());
            this.remoteAddresses = ImmutableSet.copyOf(application.getRemoteAddresses());
            this.webhooks = ImmutableSet.copyOf(application.getWebhooks());
            this.lowercaseOutput = application.isLowerCaseOutput();
            this.aliasingEnabled = application.isAliasingEnabled();
            this.membershipAggregationEnabled = application.isMembershipAggregationEnabled();
            this.cachedDirectoriesAuthenticationOrderOptimisationEnabled = application.isCachedDirectoriesAuthenticationOrderOptimisationEnabled();
            this.authenticationWithoutPasswordEnabled = application.isAuthenticationWithoutPasswordEnabled();
            this.filteringUsersWithAccessEnabled = application.isFilteringUsersWithAccessEnabled();
            this.filteringGroupsWithAccessEnabled = application.isFilteringGroupsWithAccessEnabled();
            this.createdDate = application.getCreatedDate();
            this.updatedDate = application.getUpdatedDate();
        }

        public Builder setId(final Long id) {
            this.id = id;
            return this;
        }

        public Builder setName(final String name) {
            this.name = name;
            return this;
        }

        public Builder setType(final ApplicationType type) {
            this.type = type;
            return this;
        }

        public Builder setDescription(final String description) {
            this.description = description;
            return this;
        }

        public Builder setPasswordCredential(final PasswordCredential passwordCredential) {
            this.passwordCredential = passwordCredential;
            return this;
        }

        public Builder setPermanent(final boolean permanent) {
            this.permanent = permanent;
            return this;
        }

        public Builder setActive(final boolean active) {
            this.active = active;
            return this;
        }

        public Builder setAttributes(final Map<String, String> attributes) {
            this.attributes = attributes;
            return this;
        }

        public Builder setDirectoryMappings(final List<DirectoryMapping> directoryMappings) {
            setApplicationDirectoryMappings(directoryMappings);
            return this;
        }

        public Builder setApplicationDirectoryMappings(final Collection<? extends ApplicationDirectoryMapping> directoryMappings) {
            this.directoryMappings = directoryMappings.stream()
                    .map(ImmutableApplicationDirectoryMapping::from)
                    .collect(Collectors.toList());
            return this;
        }

        public Builder setRemoteAddresses(final Set<RemoteAddress> remoteAddresses) {
            this.remoteAddresses = remoteAddresses;
            return this;
        }

        public Builder setWebhooks(Set<Webhook> webhooks) {
            this.webhooks = webhooks;
            return this;
        }

        public Builder setLowercaseOutput(final boolean lowercaseOutput) {
            this.lowercaseOutput = lowercaseOutput;
            return this;
        }

        public Builder setAliasingEnabled(final boolean aliasingEnabled) {
            this.aliasingEnabled = aliasingEnabled;
            return this;
        }

        public Builder setMembershipAggregationEnabled(boolean membershipAggregationEnabled) {
            this.membershipAggregationEnabled = membershipAggregationEnabled;
            return this;
        }

        public Builder setCreatedDate(final Date createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public Builder setUpdatedDate(final Date updatedDate) {
            this.updatedDate = updatedDate;
            return this;
        }

        public Builder setFilteringUsersWithAccessEnabled(boolean filteringUsersWithAccessEnabled) {
            this.filteringUsersWithAccessEnabled = filteringUsersWithAccessEnabled;
            return this;
        }

        public Builder setFilteringGroupsWithAccessEnabled(boolean filteringGroupsWithAccessEnabled) {
            this.filteringGroupsWithAccessEnabled = filteringGroupsWithAccessEnabled;
            return this;
        }

        public Builder setCachedDirectoriesAuthenticationOrderOptimisationEnabled(boolean cachedDirectoriesAuthenticationOrderOptimisationEnabled) {
            this.cachedDirectoriesAuthenticationOrderOptimisationEnabled = cachedDirectoriesAuthenticationOrderOptimisationEnabled;
            return this;
        }

        public Builder setAuthenticationWithoutPasswordEnabled(boolean authenticationWithoutPasswordEnabled) {
            this.authenticationWithoutPasswordEnabled = authenticationWithoutPasswordEnabled;
            return this;
        }

        /**
         * Builds the new <tt>ImmutableApplication</tt>.
         *
         * @return new <tt>ImmutableApplication</tt>
         */
        public ImmutableApplication build() {
            return new ImmutableApplication(id, name, type, description, passwordCredential, permanent, active,
                    attributes, directoryMappings, remoteAddresses, webhooks, lowercaseOutput,
                    aliasingEnabled, membershipAggregationEnabled, cachedDirectoriesAuthenticationOrderOptimisationEnabled,
                    authenticationWithoutPasswordEnabled, filteringUsersWithAccessEnabled,
                    filteringGroupsWithAccessEnabled, createdDate, updatedDate);
        }
    }

    @Override
    public String toString() {
        // intentionally omitting attributes to avoid exposing sensitive ones
        return MoreObjects.toStringHelper(this)
                .add("id", id)
                .add("name", name)
                .add("type", type)
                .add("description", description)
                .add("permanent", permanent)
                .add("active", active)
                .add("directoryMappings", directoryMappings)
                .add("remoteAddresses", remoteAddresses)
                .toString();
    }
}
