package com.atlassian.crowd.util;

import java.util.Objects;

/**
 * A possibly approximate count of the number of elements in a collection. The collection has at least {@link #getCount()}
 * elements. A count marked as exact has exactly count elements (see {@link #isExact()}. This allows APIs to give
 * indications of collection size without committing to expensive operations when an exact count requires
 * significant work.
 *
 * Provided are some example scenarios and what the results mean:
 *
 * <dl>
 * <dt>(getCount(), isExact())</dt>
 * <dd>result</dd>
 * <dt>(5, true)</dt>
 * <dd>Count of elements is exactly 5</dd>
 * <dt>(5, false)</dt>
 * <dd>Count of elements is at least 5</dd>
 * <dt>(0, false)</dt>
 * <dd>Count of elements is zero or more (could be anything)</dd>
 * </dl>
 */
public class BoundedCount {
    private final long count;
    private final boolean exact;

    /**
     * Create a bounded count that exactly represents a number.
     *
     * @param count The exact count that this represents; not a bounded number.
     * @return A BoundedCount that is exact.
     */
    public static BoundedCount exactly(long count) {
        return new BoundedCount(count, true);
    }

    /**
     * Create a BoundedCount that is at-least as big as a given number. This creates a lower bound on the provided
     * count allowing you to indicate that the collection you are counting is bigger than a given count.
     *
     * @param count A number that is at-least as big as the real count of elements in a collection.
     * @return A lower bounded count.
     */
    public static BoundedCount atLeast(long count) {
        return new BoundedCount(count, false);
    }

    /**
     * Get a bounded count from counting up to an upper bound.
     *
     * @param count             The number of elements that were actually counted.
     * @param potentialMaxCount The maximum number of elements that we counted up to.
     * @return Returns exactly the count given (if less than the upper bound) and returns at-least the upper bound otherwise.
     */
    public static BoundedCount fromCountedItemsAndLimit(long count, long potentialMaxCount) {
        return (count < potentialMaxCount) ? exactly(count) : atLeast(potentialMaxCount);
    }

    private BoundedCount(long count, boolean exactly) {
        if (count < 0)
            throw new IllegalArgumentException("Bounded count can not be instantiated with negative count: " + count);
        this.count = count;
        this.exact = exactly;
    }

    /**
     * The number of elements counted in a collection.
     *
     * @return If {@link #isExact()} returns true then it returns the exact size of the collection. Otherwise it
     * returns a lower bound of the size of the collection. The answer will always be non-negative.
     */
    public long getCount() {
        return count;
    }

    /**
     * @return True if {@link #getCount()} is known to be the exact number of elements in the collection. It will return
     * False if this count represents a lower bound.
     */
    public boolean isExact() {
        return exact;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof BoundedCount)) return false;

        BoundedCount that = (BoundedCount) o;
        return count == that.count && exact == that.exact;
    }

    @Override
    public int hashCode() {
        return Objects.hash(getCount(), isExact());
    }

    @Override
    public String toString() {
        return (exact ? "exactly" : "atLeast") + "(" + count + ")";
    }
}
