package com.atlassian.crowd.model.permission;

import com.google.common.collect.ImmutableMap;

import javax.annotation.Nullable;
import java.util.EnumSet;
import java.util.Set;

public enum UserPermission {
    ADMIN(1),

    SYS_ADMIN(2);

    private static final ImmutableMap<Integer, UserPermission> ID_TO_PERMISSION_MAP;

    static {
        ImmutableMap.Builder<Integer, UserPermission> builder = ImmutableMap.builder();
        for (UserPermission permission : values()) {
            builder.put(permission.getId(), permission);
        }
        ID_TO_PERMISSION_MAP = builder.build();
    }

    private final int id;

    private UserPermission(int id) {
        this.id = id;
    }

    public int getId() {
        return id;
    }

    /**
     * Gets all the permissions.
     */
    public static Set<UserPermission> allPermissions() {
        return EnumSet.allOf(UserPermission.class);
    }

    /**
     * Gets a permission by its id.
     *
     * IMPLEMENTATION NOTE: This method is used by the GenericEnumUserType to map from the ID stored in the database to the enumeration entry
     *
     * @param id id of the permission
     * @return the permission, or {@code null} if no permission matches the id
     */
    @Nullable
    public static UserPermission fromId(int id) {
        return ID_TO_PERMISSION_MAP.get(id);
    }

}
