package com.atlassian.crowd.manager.avatar;

import java.net.URI;

import org.apache.commons.codec.binary.Base64;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A directory may represent avatars as references to URLs or as binary blobs. This interface
 * unifies those two possibilities behind a single marker interface.
 */
public interface AvatarReference {
    /**
     * Return a URI, which may be a large <code>data:</code> URI, that points to this avatar. This is not
     * suitable for redirecting to with a Location header, but is suitable for embedding
     * directly in an HTML page.
     */
    URI toUri();

    /**
     * A reference to an avatar available through a user's browser.
     */
    public static class UriAvatarReference implements AvatarReference {
        private final URI uri;

        public UriAvatarReference(URI uri) {
            this.uri = checkNotNull(uri);
        }

        public URI getUri() {
            return uri;
        }

        @Override
        public URI toUri() {
            return uri;
        }
    }

    /**
     * An image represented as a blob of bytes along with a content type.
     */
    public static class BlobAvatar implements AvatarReference {
        private final String contentType;
        private final byte[] content;

        public BlobAvatar(String contentType, byte[] content) {
            this.contentType = checkNotNull(contentType);
            this.content = checkNotNull(content);
        }

        public String getContentType() {
            return contentType;
        }

        public byte[] getContent() {
            return content;
        }

        @Override
        public URI toUri() {
            return URI.create("data:" + getContentType() + ";base64," + Base64.encodeBase64String(getContent()));
        }
    }
}
