(function($) {

    var EDIT_MACRO_BUTTON_EDIT_CLASS = "macro-placeholder-property-panel-edit-button";
    // css class for both label and input representations of the title
    var INPUT_ELEMENT_CLASSES = "status-macro-title first last editable";
    var ENTER_KEY = 13;
    var ESCAPE_KEY = 27;

    var STATUS_MACRO_NAME = "status";
    // property-panel button ids from the plugin descriptor, assumes that the parameter names equal the button ids
    var STATUS_MACRO_COLORS = [ "Grey", "Red", "Yellow", "Green", "Blue"];
    var STATUS_MACRO_COLOR_DEFAULT = STATUS_MACRO_COLORS[0];

    var deferredUpdate;
    var macroNode, macroParameters, $inputTitleElement;


    /**
     * Check if the macro has been updated
     * @param newParameters
     * @returns {boolean}
     */
    function hasMacroChanged() {

        updateParameters();

        var originalMacroParameters = Confluence.MacroParameterSerializer.deserialize($(macroNode).attr("data-macro-parameters")),
            hasTitleUpdated  = false,
            hasColourUpdated = false;

        if(originalMacroParameters && macroParameters) {
            hasTitleUpdated = originalMacroParameters.title !== macroParameters.title;
            hasColourUpdated = originalMacroParameters.colour !== macroParameters.colour;
        }

        return hasTitleUpdated || hasColourUpdated;
    }

    /**
     * Binds macroParameters with values from the UI
     */
    function updateParameters() {
        if ($inputTitleElement) {
            var isDefaultMacroTitle = (getMacroColor(macroParameters) == $inputTitleElement.val());
            var isEmpty = ($inputTitleElement.val() === "");
            if (isDefaultMacroTitle || isEmpty) {
                delete macroParameters.title;
            }
            else {
                macroParameters.title = $inputTitleElement.val();
            }
        }
    }

    /**
     * Calls the server to get the updated node and update the DOM asynchronously.
     */
    function updateMacro() {

        if (deferredUpdate && deferredUpdate.state && deferredUpdate.state() === "pending"){
            AJS.log("Macro update already in progress.");
            return; // request not completed
        }

        updateParameters();

        var macroRenderRequestParameters = {
            contentId : Confluence.Editor.getContentId(),
            macro : {
                name : STATUS_MACRO_NAME,
                params : macroParameters,
                defaultParameterValue : "",
                body : ""
            }
        };

        // onGoingRequest is a jQuery deferred object to be fulfilled when the operation is completed
        deferredUpdate = tinymce.confluence.MacroUtils.insertMacro(macroRenderRequestParameters, macroNode);

        function selectNodeAndBookmark(macroNode) {
            AJS.Rte.getEditor().selection.select(macroNode);
            AJS.Rte.BookmarkManager.storeBookmark();
            AJS.Rte.getEditor().selection.collapse();
        }

        deferredUpdate
            .done(function(newNode, newNodeMarkup) {
                macroNode = newNode;
                selectNodeAndBookmark(macroNode);
            })
            .fail(function(error) {
                AJS.logError("Failed to load status macro - " + error);
            });
    }

    /**
     * Retrieves the current color from the parameters or returns the default.
     * @param {object} macroParameters whose color and colour property will be read.
     * @returns {string} a color from STATUS_MACRO_COLORS such as "Red"
     */
    function getMacroColor (macroParameters) {
        return ((macroParameters.colour || macroParameters.color) || STATUS_MACRO_COLOR_DEFAULT);
    }

    function cancelEdit() {
        $inputTitleElement = null;
        return true;
    }

    /**
     * Builds the actual input box element
     */
    function makeTitleInputElement(macroParameters) {
        var $inputTitleElement = $("<input/>");
        $inputTitleElement.attr("class", INPUT_ELEMENT_CLASSES);

        $inputTitleElement.keyup(function(e) {
            if (e.keyCode === ENTER_KEY) {
                AJS.Confluence.PropertyPanel.destroy();
                // if its an enter key we want to prevent the default action since
                // we want the cursor right after the macro.
                return false;
            }
            return true;
        });

        $inputTitleElement.keydown(function(e) {
            if (e.keyCode === ESCAPE_KEY) {
                return cancelEdit();
            }
            return true;
        });

        return $inputTitleElement;
    }

    /**
     * Replaces the title element with an editable input text with event handlers.
     */
    function clickTitleInputElement($titleButtonElement) {
        $inputTitleElement = makeTitleInputElement(macroParameters);
        $inputTitleElement.val($titleButtonElement.find("span.panel-button-text").text());
        $titleButtonElement.replaceWith($inputTitleElement);
        $inputTitleElement.select().focus();
    }

    /**
     * Register event handlers for click events on color buttons
     */
    function registerColorButtonsEventHandlers() {

        $.each(STATUS_MACRO_COLORS, function (idx, colour) {

            var onColorButtonClick = function (e, macroNode) {
                macroParameters.colour = colour;
                AJS.Confluence.PropertyPanel.destroy();
            };

            AJS.Confluence.PropertyPanel.Macro.registerButtonHandler(colour, onColorButtonClick, STATUS_MACRO_NAME);
        });
    }

    function getInitialTitle(macroParameters) {
        return macroParameters.title || getMacroColor(macroParameters);
    }

    /**
     * Creates the label/button title element. When clicked, it gets converted in an input text box.
     * @param currentTitle
     * @returns {{className: string, text: *, click: Function}}
     */
    function createTitleButtonElement(currentTitle) {
        return {
            className: INPUT_ELEMENT_CLASSES,
            text: AJS.escapeHtml(currentTitle),
            click: clickTitleInputElement
        };
    }

    /**
     * Gets the edit button among the collection of available buttons, checking its CSS class (not ideal)
     * @param buttons
     * @returns {*}
     */
    function getEditButton(buttons) {
        for (var i = 0; i < buttons.length; i++) {
            if (buttons[i].className.indexOf(EDIT_MACRO_BUTTON_EDIT_CLASS) > -1){
                return buttons[i];
            }
        }
        return null;
    }

    /**
     * Override the edit button to wait for the updateMacro's promise to be resolved before opening the macrobrowser
     * dialog.
     *
     * @param buttons
     */
    function overrideEditButton(buttons) {
        var editButton = getEditButton(buttons);

        var editClickAction = function (node) {
            tinymce.confluence.macrobrowser.editMacro($(node));
        };

        editButton.click = function () {
            AJS.Confluence.PropertyPanel.destroy();

            if (!hasMacroChanged()) {
                editClickAction(macroNode);
            }
            else {
                // deferredUpdate is populated by calling destroy() if the macro has changed, since destroy() calls updateMacro, which
                // sets deferredUpdate.
                // This should probably be less confusing...
                deferredUpdate.always(editClickAction);
            }
        };
    }

    AJS.Confluence.PropertyPanel.Macro.registerInitHandler(function (initMacroNode, buttons, options) {
        macroNode = initMacroNode;
        overrideEditButton(buttons);
        macroParameters = Confluence.MacroParameterSerializer.deserialize($(macroNode).attr("data-macro-parameters"));
        $inputTitleElement = null;

        // insert title button after the first element = 'Edit'
        buttons.splice(1, 0, createTitleButtonElement(getInitialTitle(macroParameters)));
    }, "status");

    registerColorButtonsEventHandlers();

    AJS.bind("destroyed.property-panel", function(e){
        if (hasMacroChanged()){
            updateMacro();
        }
    });

})(AJS.$);
