define('confluence/share-page/popup/setup-share-link', [
    'jquery',
    'confluence/share-page/service/analytics-service',
    'confluence/share-page/util/show-message'
], function ($,
             analyticsService,
             showMessage) {

    /**
     * Resolves the link. If it's a function, it will evaluate it, otherwise
     * it will just return it if it's a string, or return window.location if it's not
     * either of those things.
     * @param {function|string} link
     * @private
     */
    function _resolveLink(link) {
        if (typeof link === 'function') {
            return $.when(link());
        }
        if (typeof link === 'string') {
            return $.when(link);
        }
        return $.when(window.location);
    }

    /**
     * Adds analytics parameters to link. Returns the new link with query
     * parameters concatenated.
     * @param {string} link
     * @private
     */
    function _addAnalyticsToLink(link) {
        if (("" + link).indexOf('resumedraft.action') === -1) {
            return link;
        }

        var analyticsQueryParamString = 'src=shareui' +
            '&src.shareui.timestamp=' + (new Date()).getTime();

        var firstChar;
        if (window.location.search.length === 0) {
            firstChar = '?';
        } else {
            firstChar = '&';
        }

        return link + firstChar + analyticsQueryParamString;
    }

    /**
     * Sets up the Share Link and button to copy the link to clipboard
     * @param $container
     * @param parameters        The dialog's parameters passed-through
     */
    return function setupShareLink($container, parameters) {
        var $row = $container.find('.share-copy-link');
        var $shareLink = $row.find('input');
        var $copyButton = $row.find('button');
        // Don't set it up if it doesn't exist
        if (!$shareLink.length) {
            return;
        }

        // Set the default link, then get the real link and update it;
        $shareLink.val(window.location);
        _resolveLink(parameters.link).then(function (link) {
            link = _addAnalyticsToLink(link);
            $shareLink.val(link);
        });

        $shareLink.on('click focus',
            /**
             * Select all the contents of the input field on click
             */
            function (e) {
                setTimeout(function () {
                    _selectLink();
                }, 0);
                e.preventDefault();
            });

        $shareLink.focus(function () {
            analyticsService.publish(analyticsService.SHARE_LINK_CLICKED, parameters.entityId, parameters.shareType);
        });
        $copyButton.click(function () {
            analyticsService.publish(analyticsService.SHARE_LINK_COPY_CLICKED, parameters.entityId, parameters.shareType);
            _copy();
        });
        $shareLink.dblclick(function () {
            analyticsService.publish(analyticsService.SHARE_LINK_DOUBLE_CLICKED, parameters.entityId, parameters.shareType);
            _copy();
        });

        $shareLink.blur(function () {
            $shareLink.removeData('selected');
        });

        $(document).off('copy.share-link').on('copy.share-link', function () {
            if (!$shareLink.data('selected')) {
                return;
            }
            analyticsService.publish(analyticsService.SHARE_LINK_COPIED, parameters.entityId, parameters.shareType);
            showMessage($row, 'copied', parameters, false, function () {
                $copyButton.prop('disabled', true);
            }, function () {
                $copyButton.prop('disabled', false);
            });
        });


        /**
         * Select all the contents of the input field when the mouse
         * is there
         */
        function _selectLink() {
            $shareLink.select();
            $shareLink.data('selected', true);
        }

        /**
         * Copies the link to the clipboard
         */
        function _copy() {
            _selectLink();
            document.execCommand('copy');
        }
    }
});