define('confluence/share-page/form/submit', [
    'ajs',
    'jquery',
    'confluence/legacy',
    'confluence/meta',
    'confluence/share-page/service/analytics-service',
    'confluence/share-page/util/find-recipients',
    'confluence/share-page/util/resolve-entity-id',
    'confluence/message-controller'
], function (AJS,
             $,
             Confluence,
             Meta,
             analyticsService,
             findRecipients,
             resolveEntityId,
             MessageController) {
    var COPY_OPTION_INVITE = 'invite';

    /**
     * Generates the Ajax request data for submitting the share request
     * @param $contents of the popup
     * @param entityId Id of the entity to send
     * @param contentType Specifies a custom content type. Otherwise, the Meta.content-type will be used
     * @returns {{users: Array, emails: Array, groups: Array, note: string, entityId: (V|*)}}
     * @private
     */
    function _generateRequest($contents, entityId, contentType) {
        var users = findRecipients.findUsers($contents);
        var emails = findRecipients.findEmails($contents);
        var groups = findRecipients.findGroups($contents);
        var $note = $contents.find("#share-note-input");

        return {
            users: users,
            emails: emails,
            groups: groups,
            note: $note.val(),
            entityId: entityId,
            entityType: contentType || Meta.get('content-type'),
            contextualPageId: Meta.get('content-id')
        };
    }

    function _enableControls($contents) {
        $("button,input,textarea", $contents).removeAttr("disabled");
        $(".submit.button", $contents).removeClass('disabled').removeAttr('aria-disabled'); // to track click events
    }

    function _disableControls($contents) {
        $("button,input,textarea", $contents).attr("disabled", "disabled");
        $(".submit.button", $contents).attr('disabled', false).addClass('disabled').attr('aria-disabled', true); // to track click events
    }

    /**
     * Shows the flag to display after the operation completes or fails.
     * @param type
     * @param body
     * @private
     */
    function _setFlag(type, body) {
        var escapedBody = AJS.escapeEntities(body);
        switch (type) {
            case 'success' :
                AJS.flag({
                    type: 'success',
                    body: escapedBody,
                    close: 'auto'
                });
                break;
            /*
            Shouldn't be reached now that MessageController is being used.
             */
            case 'error':
                AJS.flag({
                    type: 'error',
                    body: escapedBody
                });

        }
    }

    function _oneInviteeSuccessMessage(copyOption, name) {
        return (copyOption === COPY_OPTION_INVITE
            ? AJS.I18n.getText('share.invite.flag.sent.to.one', name)
            : AJS.I18n.getText('share.flag.sent.to.one', name));
    }

    function _twoInviteeSuccessMessage(copyOption, name) {
        return (copyOption === COPY_OPTION_INVITE
            ? AJS.I18n.getText('share.invite.flag.sent.to.two', name)
            : AJS.I18n.getText('share.flag.sent.to.two', name));
    }

    function _moreInviteesSuccessMessage(copyOption, name, howManyMore) {
        return (copyOption === COPY_OPTION_INVITE
            ? AJS.I18n.getText('share.invite.flag.sent.to.more', name, howManyMore)
            : AJS.I18n.getText('share.flag.sent.to.more', name, howManyMore));
    }

    /**
     * Submit handler for the share request
     * @param $contents of the dialog
     * @param trigger DOM element trigger of this dialog
     * @param doHidePopup Callback function that hides the popup once the request is successful
     * @param {{entityIdFunc: {function|string|number},
     *          successText: {string},
     *          errorText: {string}
     *          }} parameters Parameters used in the templates
     */
    return function submit($contents, trigger, doHidePopup, parameters) {
        var numSelectedRecipients = findRecipients.findAllRecipients($contents).length;
        if (numSelectedRecipients === 0) {
            return false;
        }
        var $icon = $contents.find(".progress-messages-icon");

        _disableControls($contents);
        $icon.spin();
        $icon.css("position", "absolute").css("top", "20px");

        var request = _generateRequest($contents, resolveEntityId(parameters.entityId), parameters.contentType);

        $.ajax({
            type: "POST",
            contentType: "application/json; charset=utf-8",
            url: Confluence.getContextPath() + "/rest/share-page/latest/share",
            data: JSON.stringify(request),
            dataType: "text",
            success: function () {
                $icon.spinStop();
                var name = $contents.find('.recipients').find('li').first().text();
                var flagMessage;
                switch (numSelectedRecipients) {
                    case 1:
                        flagMessage = _oneInviteeSuccessMessage(parameters.copyOption, name);
                        break;
                    case 2:
                        flagMessage = _twoInviteeSuccessMessage(parameters.copyOption, name);
                        break;
                    default:
                        flagMessage = _moreInviteesSuccessMessage(parameters.copyOption, name, numSelectedRecipients - 1);
                }
                _setFlag('success', flagMessage);
                doHidePopup(true);
            },
            error: function (data, status) {
                analyticsService.publish(analyticsService.SHARE_ERROR, parameters.entityId, parameters.shareType);
                $icon.spinStop();
                MessageController.showError(MessageController.parseError(data), MessageController.Location.FLAG);
                _enableControls($contents, parameters.errorText)
            }
        });

        return false;
    }
});