define("confluence-user-profile/avatar-picker/image-upload-and-crop", [
    "jquery",
    "underscore",
    "confluence-user-profile/avatar-picker/canvas-cropper",
    "confluence-user-profile/avatar-picker/client-file-iframe-uploader",
    "confluence-user-profile/avatar-picker/client-file-reader",
    "confluence-user-profile/avatar-picker/drag-drop-file-target",
    "confluence-user-profile/avatar-picker/image-explorer",
    "confluence-user-profile/avatar-picker/text-util",
    "confluence-user-profile/avatar-picker/upload-interceptor"
],
/**
 * @tainted AJS.I18n
 */
function(
    $,
    _,
    CanvasCropper,
    ClientFileIframeUploader,
    ClientFileReader,
    DragDropFileTarget,
    ImageExplorer,
    TextUtil,
    UploadInterceptor
) {
    "use strict";

    function ImageUploadAndCrop($container, opts) {
        if (!ImageUploadAndCrop.isSupported()) {
            AJS.messages.error({
                body: "This browser doesn't support ImageUploadAndCrop."
            });
        }
        this.init.apply(this, arguments);
    }

    ImageUploadAndCrop.isSupported = function() {
        return CanvasCropper.isSupported();
    };

    ImageUploadAndCrop.prototype.defaults = {
        HiDPIMultiplier: 2,  //The canvas crop size is multiplied by this to support HiDPI screens
        dragDropUploadPrompt: AJS.I18n.getText('sidebar.avatar.picker.drag.image'),
        onImageUpload: function() {},
        onImageUploadError: function() {},
        onCrop: function() {},
        outputFormat: 'image/jpeg',
        outputQuality: 0.9,
        fallbackUploadOptions: {},
        initialScaleMode: ImageExplorer.scaleModes.containAndFill,
        scaleMax: 1,
        fileSizeLimit: 10 * 1024 * 1024, //10MB
        maxImageDimension: 4000 //In pixels
    };

    ImageUploadAndCrop.prototype.init = function($container, opts) {
        this.options = _.extend({}, this.defaults, opts);
        this.$container = $container;

        _.bindAll(this, "crop", "resetState", "_onFileProcessed", "setImageSrc", "validateImageResolution", "_onFilesError",
            "_onFileError", "_resetFileUploadField", "_onErrorReset", "_rotateImage");

        this.imageExplorer = new ImageExplorer(this.$container.find('.profile-pic-explorer-container'), {
            initialScaleMode: this.options.initialScaleMode,
            scaleMax: this.options.scaleMax,
            onErrorReset: this._onErrorReset
        });

        if (ClientFileReader.isSupported()) {
            this.clientFileReader = new ClientFileReader({
                onRead: this._onFileProcessed,
                onError: this._onFilesError,
                fileTypeFilter: ClientFileReader.typeFilters.imageWeb,
                fileCountLimit: 1,
                fileSizeLimit: this.options.fileSizeLimit
            });

            //drag drop uploading is only possible in browsers that support the fileReaderAPI
            this.dragDropFileTarget = new DragDropFileTarget(this.imageExplorer.get$ImageView(), {
                uploadPrompt: this.options.dragDropUploadPrompt,
                clientFileHandler: this.clientFileReader
            });
        } else {
            //Fallback for older browsers.

            this.$container.addClass("filereader-unsupported");

            var fallbackOptions = _.extend({
                onUpload: this._onFileProcessed,
                onError: this._onFileError
            }, this.options.fallbackUploadOptions);

            this.clientFileReader = new ClientFileIframeUploader(fallbackOptions);
        }

        this.uploadIntercepter = new UploadInterceptor(this.$container.find('.image-upload-field'), {
            replacementEl: this.$container.find('.image-upload-field-replacement'),
            clientFileHandler: this.clientFileReader
        });

        var mask = this.imageExplorer.get$Mask();

        this.canvasCroppper = new CanvasCropper(
            mask.width() * this.options.HiDPIMultiplier,
            mask.height() * this.options.HiDPIMultiplier,
            {
                outputFormat: this.options.outputFormat,
                outputQuality: this.options.outputQuality
            }
        );

        this.options.cropButton && $(this.options.cropButton).click(this.crop);
    };

    ImageUploadAndCrop.prototype.crop = function() {
        var croppedDataURI = "";
        if (!this.imageExplorer.$container.hasClass('empty')) {
            var cropProperties = this.imageExplorer.getMaskedImageProperties();
            croppedDataURI = this.canvasCroppper.cropToDataURI(
                    this.imageExplorer.get$SourceImage()[0],
                    cropProperties.maskedAreaImageX,
                    cropProperties.maskedAreaImageY,
                    cropProperties.maskedAreaWidth,
                    cropProperties.maskedAreaHeight
            );
        }

        _.isFunction(this.options.onCrop) && this.options.onCrop(croppedDataURI);
    };

    ImageUploadAndCrop.prototype.resetState = function() {
        this.imageExplorer.clearError();
        this._resetFileUploadField();
    };

    ImageUploadAndCrop.prototype.saveState = function() {
        if (this.imageExplorer.$container.hasClass('empty')) {
            return;
        }
        var $sourceImage = this.imageExplorer.get$SourceImage();
        if ($sourceImage.length) {
            this.imageState = {
                top: $sourceImage.css('top'),
                left: $sourceImage.css('left'),
                marginTop: $sourceImage.css('margin-top'),
                marginLeft: $sourceImage.css('margin-left'),
                zoom: $('.image-explorer-scale-slider').val()
            }
        }
    };

    ImageUploadAndCrop.prototype.restoreState = function() {
        if (this.imageExplorer.$container.hasClass('empty') || !this.imageState) {
            return;
        }
        var $sourceImage = this.imageExplorer.get$SourceImage();
        if ($sourceImage.length) {
            this.imageExplorer.$scaleSlider.val(this.imageState.zoom).trigger('change');
            $sourceImage.css({
                top: this.imageState.top,
                left: this.imageState.left,
                "margin-top": this.imageState.marginTop,
                "margin-left": this.imageState.marginLeft
            });
        }
    };

    ImageUploadAndCrop.prototype._onFileProcessed = function(imageSrc) {
        if (imageSrc) {

            var validateImagePromise = this.validateImage(imageSrc);
            var self = this;
            validateImagePromise
                .done(function() {
                    if (!isNaN(self.options.maxImageDimension)) {
                        var validateResolutionPromise = self.validateImageResolution(imageSrc);

                        validateResolutionPromise
                            .done(function(imageWidth, imageHeight) {
                                self.setImageSrc(imageSrc);
                            })
                            .fail(function(imageWidth, imageHeight) {
                                self._onFileError(AJS.I18n.getText('user.avatar.picker.error.resolution', imageWidth, imageHeight, self.options.maxImageDimension));
                            });
                    } else {
                        // If imageResolutionMax isn't valid, skip the validation and just set the image src.
                        self.setImageSrc(imageSrc);
                    }
                })
                .fail(function() {
                    self._onFileError(AJS.I18n.getText('user.avatar.picker.error.file.type'));
                })
        } else {
            this._onFileError();
        }
    };

    ImageUploadAndCrop.prototype.validateImage = function(imageSrc) {
        var validatePromise = $.Deferred(),
                tmpImage = new Image();

        tmpImage.onerror = function() {
            validatePromise.reject();
        };

        tmpImage.onload = function() {
            validatePromise.resolve();
        };

        tmpImage.src = imageSrc;

        return validatePromise;
    };

    ImageUploadAndCrop.prototype.setImageSrc = function(imageSrc) {
        this._rotateImage(imageSrc, _.bind(function(rotatedImageSrc) {
            this.imageExplorer.setImageSrc(rotatedImageSrc);
            _.isFunction(this.options.onImageUpload) && this.options.onImageUpload(rotatedImageSrc);
            this._resetFileUploadField();
        }, this));
    };

    ImageUploadAndCrop.prototype._rotateImage = function(imageSrc, callback) {
        if (!_.isObject(window.EXIF)) {
            //fallback to original if EXIF library is not initialized.
            callback(imageSrc);
            return;
        }
        var img = new Image();

        img.onload = function() {
            //load the EXIF metadata which would be stored to the data attribute of img
            window.EXIF.getData(img);
            var exifOrientation = window.EXIF.getTag(img, 'Orientation');
            if (_.isUndefined(exifOrientation)) {
                //fallback to default if EXIF metadata does not exist.
                callback(imageSrc);
                return;
            }

            var width = img.width,
                height = img.height,
                canvas = document.createElement('canvas'),
                ctx = canvas.getContext("2d");

            // set proper canvas dimensions before transform & export
            if ([5,6,7,8].indexOf(exifOrientation) > -1) {
                canvas.width = height;
                canvas.height = width;
            } else {
                canvas.width = width;
                canvas.height = height;
            }

            // transform context before drawing image
            // The six parameters of <b>ctx</b> represents horizontal scaling, horizontal skewing, vertical scaling,
            // vertical skewing, horizontal moving and vertical moving.
            // @see https://developer.mozilla.org/en-US/docs/Web/API/CanvasRenderingContext2D/transform
            switch (exifOrientation) {
                case 2: ctx.transform(-1, 0, 0, 1, width, 0); break;
                case 3: ctx.transform(-1, 0, 0, -1, width, height ); break;
                case 4: ctx.transform(1, 0, 0, -1, 0, height ); break;
                case 5: ctx.transform(0, 1, 1, 0, 0, 0); break;
                case 6: ctx.transform(0, 1, -1, 0, height , 0); break;
                case 7: ctx.transform(0, -1, -1, 0, height , width); break;
                case 8: ctx.transform(0, -1, 1, 0, 0, width); break;
                default: ctx.transform(1, 0, 0, 1, 0, 0);
            }

            // draw image
            ctx.drawImage(img, 0, 0);

            // export base64
            callback(canvas.toDataURL());
        };
        img.src = imageSrc;
    };

    ImageUploadAndCrop.prototype.validateImageResolution = function(imageSrc) {
        var validatePromise = $.Deferred(),
            tmpImage = new Image(),
            self = this;

        tmpImage.onload = function() {
            if (this.naturalWidth > self.options.maxImageDimension ||  this.naturalHeight > self.options.maxImageDimension) {
                validatePromise.reject(this.naturalWidth, this.naturalHeight);
            } else {
                validatePromise.resolve(this.naturalWidth, this.naturalHeight);
            }
        };

        tmpImage.src = imageSrc;

        return validatePromise;
    };

    ImageUploadAndCrop.prototype._onFilesError = function(invalidFiles) {
        // Work out the most appropriate error to display. Because drag and drop uploading can accept multiple files and we can't restrict this,
        // it's not an all or nothing situation, we need to try and find the most correct file and base the error on that.
        // If there was at least 1 valid file, then this wouldn't be called, so we don't need to worry about files rejected because of the fileCountLimit

        if (invalidFiles && invalidFiles.bySize && invalidFiles.bySize.length) {
            //Some image files of the correct type were filtered because they were too big. Pick the first one to use as an example.
            var file = _.first(invalidFiles.bySize);
            this._onFileError(AJS.I18n.getText('user.avatar.picker.error.file.size', TextUtil.abbreviateText(file.name, 50),
                    TextUtil.formatSizeInBytes(file.size), TextUtil.formatSizeInBytes(this.options.fileSizeLimit)));
        } else {
            //No files of the correct type were uploaded. The default error message will cover this.
            this._onFileError();
        }
    };

    ImageUploadAndCrop.prototype._onFileError = function(error) {
        AJS.log('onFileError', error);

        var title = AJS.I18n.getText('user.avatar.picker.error.title'),
            contents = error || AJS.I18n.getText('user.avatar.picker.error.file.type');

        this.imageExplorer.showError(title, contents);
        this._resetFileUploadField();
        _.isFunction(this.options.onImageUploadError) && this.options.onImageUploadError(error);
    };

    ImageUploadAndCrop.prototype._resetFileUploadField = function() {
        //clear out the fileUpload field so the user could select the same file again to "reset" the imageExplorer
        var form = this.$container.find("#image-upload-and-crop-upload-field").prop('form');
        form && form.reset();
    };

    ImageUploadAndCrop.prototype._onErrorReset = function(imgSrc) {
        //If we have a valid image after resetting from the error, notify the calling code.
        if (imgSrc) {
            _.isFunction(this.options.onImageUpload) && this.options.onImageUpload(imgSrc);
        }
    };

    return ImageUploadAndCrop;
});