package com.atlassian.confluence.plugins.team;

import com.atlassian.confluence.plugins.createcontent.api.events.SpaceBlueprintCreateEvent;
import com.atlassian.confluence.security.SpacePermission;
import com.atlassian.confluence.security.SpacePermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.renderer.v2.components.HtmlEscaper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.List;

import static java.util.Arrays.asList;

@Component
public class SpaceBlueprintEventListener {

    private static final int MAXIMUM_TWO_LINES = 14;
    private static final int MINIMUM_TWO_LINES = 6;

    private final EventPublisher eventPublisher;
    private final UserAccessor userAccessor;
    private final SpacePermissionManager spacePermissionManager;

    @Autowired
    public SpaceBlueprintEventListener(@ComponentImport EventPublisher eventPublisher,
                                       @ComponentImport UserAccessor userAccessor,
                                       @ComponentImport SpacePermissionManager spacePermissionManager) {
        this.eventPublisher = eventPublisher;
        this.userAccessor = userAccessor;
        this.spacePermissionManager = spacePermissionManager;
    }

    @PostConstruct
    public void initialise() {
        this.eventPublisher.register(this);
    }

    @PreDestroy
    public void teardown() {
        this.eventPublisher.unregister(this);
    }

    @EventListener
    public void onSpaceBlueprintCreate(SpaceBlueprintCreateEvent event) {
        if (!TeamSpaceConstants.TEAM_SPACE_COMPLETE_KEY.equals(event.getSpaceBlueprint().getModuleCompleteKey())) {
            return;
        }

        // Add members as space watchers
        String members = (String) event.getContext().get("members");

        final Space space = event.getSpace();
        String[] userNames = members.split(",");
        String teamGrid = generateTeamGrid(userNames);
        grantPermissionsToTeamMembers(asList(userNames), space);

        event.getContext().put("team", teamGrid);
    }

    private void grantPermissionsToTeamMembers(List<String> usernames, Space space) {
        for (String username : usernames) {
            final ConfluenceUser user = userAccessor.getUserByName(username);
            if (user != null) {
                // grant team members all permissions in the space
                for (String permission : SpacePermission.GENERIC_SPACE_PERMISSIONS) {
                    spacePermissionManager.savePermission(SpacePermission.createUserSpacePermission(permission, space, user));
                }
            }
        }
    }

    String generateTeamGrid(final String... userNames) {
        StringBuilder sb = new StringBuilder();

        int colIdx;
        final int MAX_COL_NUM;

        int numUsers = userNames.length;

        if (numUsers < MINIMUM_TWO_LINES) {
            MAX_COL_NUM = numUsers;
        } else if (numUsers < MAXIMUM_TWO_LINES) {
            MAX_COL_NUM = (int) Math.ceil(numUsers / 2f);
        } else {
            MAX_COL_NUM = 7;
        }
        colIdx = MAX_COL_NUM;

        for (String username : userNames) {
            final ConfluenceUser user = userAccessor.getUserByName(username);
            if (user != null) {
                // Row beginning
                if (colIdx == MAX_COL_NUM) {
                    sb.append("<tr>\n");
                }

                // Cell content
                fillUserTemplate(sb, user);

                // Row end
                colIdx--;
                if (colIdx == 0) {
                    sb.append("</tr>\n");
                    colIdx = MAX_COL_NUM;
                }
            }
        }
        if (numUsers > MINIMUM_TWO_LINES && colIdx < MAX_COL_NUM) {
            while (colIdx-- > 0) {
                sb.append("<td></td>\n");
            }
            sb.append("</tr>\n");
        }

        return sb.toString();
    }

    static void fillUserTemplate(final StringBuilder sb, final ConfluenceUser user) {
        sb.append("<td><p style=\"text-align: center;\">");
        //sb.append(String.format("<ac:macro ac:name=\"profile-picture\"><ac:parameter ac:name=\"User\">%1$s</ac:parameter></ac:macro>",
        sb.append(String.format("<ac:structured-macro ac:name=\"profile-picture\"><ac:parameter ac:name=\"User\"><ri:user ri:userkey=\"%1$s\" /></ac:parameter></ac:structured-macro>",
                HtmlEscaper.escapeAll(user.getKey().getStringValue(), false)));
        sb.append("</p><p style=\"text-align: center;\">");
        sb.append(String.format("<strong><a href=\"mailto:%1$s\">%2$s</a></strong>",
                HtmlEscaper.escapeAll(user.getEmail(), false),
                HtmlEscaper.escapeAll(user.getFullName(), false)));
        sb.append("</p></td>\n");
    }
}
