package com.atlassian.confluence.plugins.documentation;

import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.plugins.createcontent.api.events.SpaceBlueprintCreateEvent;
import com.atlassian.confluence.plugins.ia.SidebarLinkCategory;
import com.atlassian.confluence.plugins.ia.rest.SidebarLinkBean;
import com.atlassian.confluence.plugins.ia.service.SidebarLinkService;
import com.atlassian.confluence.plugins.ia.service.SidebarService;
import com.atlassian.confluence.rpc.NotPermittedException;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Map;

@Component
public class CreateSpaceEventListener {

    private static final String DOCUMENTATION_SPACE_COMPLETE_KEY = "com.atlassian.confluence.plugins.confluence-space-blueprints:documentation-space-blueprint";

    private final SidebarService sidebarService;
    private final SidebarLinkService sidebarLinkService;
    private final I18NBeanFactory i18NBeanFactory;
    private final LocaleManager localeManager;
    private final EventPublisher eventPublisher;

    @Autowired
    public CreateSpaceEventListener(@ComponentImport SidebarService sidebarService,
                                    @ComponentImport SidebarLinkService sidebarLinkService,
                                    @ComponentImport I18NBeanFactory i18NBeanFactory,
                                    @ComponentImport LocaleManager localeManager,
                                    @ComponentImport EventPublisher eventPublisher) {
        this.sidebarService = sidebarService;
        this.sidebarLinkService = sidebarLinkService;
        this.i18NBeanFactory = i18NBeanFactory;
        this.localeManager = localeManager;
        this.eventPublisher = eventPublisher;
    }

    @PostConstruct
    public void initialise() {
        this.eventPublisher.register(this);
    }

    @PreDestroy
    public void teardown() {
        this.eventPublisher.unregister(this);
    }

    @EventListener
    public void onSpaceBlueprintCreate(final SpaceBlueprintCreateEvent event) throws NotPermittedException {
        if (!event.getSpaceBlueprint().getModuleCompleteKey().equals(DOCUMENTATION_SPACE_COMPLETE_KEY)) {
            return;
        }

        // add our spacekey to the context so we can shove it into the homepage template
        Map<String, Object> context = event.getContext();
        Space space = event.getSpace();
        context.put("spaceKey", space.getKey());

        // create a var for the i18n'ed link to the readme page
        I18NBean i18NBean = i18NBeanFactory.getI18NBean(localeManager.getLocale(AuthenticatedUserThreadLocal.get()));
        String pageTitle = i18NBean.getText("confluence.blueprints.space.documentation.making-a-template.name");
        context.put("makingATemplateLink", "<ri:page ri:content-title=\"" + pageTitle + "\" />");

        String spaceKey = space.getKey();
        // this space wants a minimalist sidebar
        sidebarService.setOption(spaceKey, SidebarService.NAV_TYPE_OPTION, SidebarService.PAGE_TREE_NAV_TYPE);
        sidebarService.setOption(spaceKey, SidebarService.QUICK_LINKS_STATE_OPTION, SidebarService.QUICK_LINKS_HIDE_STATE);
        for (SidebarLinkBean link : sidebarLinkService.getLinksForSpace(SidebarLinkCategory.MAIN, spaceKey, false)) {
            sidebarLinkService.hide(link.getId());
        }
    }
}
