package com.atlassian.confluence.plugins.team;

import com.atlassian.confluence.core.DataSourceFactory;
import com.atlassian.confluence.core.PluginDataSourceFactory;
import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.links.SimpleLink;
import com.atlassian.confluence.mail.notification.NotificationManager;
import com.atlassian.confluence.mail.notification.listeners.NotificationTemplate;
import com.atlassian.confluence.mail.template.PreRenderedMailNotificationQueueItem;
import com.atlassian.confluence.plugin.descriptor.mail.NotificationContext;
import com.atlassian.confluence.plugin.descriptor.mail.NotificationRenderManager;
import com.atlassian.confluence.plugins.createcontent.api.events.SpaceBlueprintHomePageCreateEvent;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.core.task.MultiQueueTaskManager;
import com.atlassian.core.util.PairType;
import com.atlassian.event.api.EventListener;
import com.atlassian.fugue.Maybe;
import com.atlassian.mail.queue.MailQueueItem;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;
import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class SpaceHomePageEventListener
{
    private static final String EMAIL_RESOURCE_KEY = "com.atlassian.confluence.plugins.confluence-email-resources";
    private static final String TEAM_SPACE_EMAIL_RESOURCES = "team-space-email-resources";
    private static final String TEAM_SPACE_EMAIL_COMPLETE_KEY = TeamSpaceConstants.TEAM_SPACE_MODULE_KEY + ":" + TEAM_SPACE_EMAIL_RESOURCES;
    private static final String TEAM_SPACE_EMAIL_SOY_TEMPLATE = "Confluence.Templates.Team.Space.Notifications.teamSpaceEmail.soy";

    private static final String TEAM_SPACE_ICON_KEY = "team-space-icon";
    private static final String TEAM_SPACE_TYPE = "team-space";

    private List<DataSource> iconResources = Lists.newArrayList();

    private final UserAccessor userAccessor;
    private final NotificationManager notificationManager;
    private final DataSourceFactory dataSourceFactory;
    private final I18NBeanFactory i18NBeanFactory;
    private final LocaleManager localeManager;
    private final MultiQueueTaskManager taskManager;
    private final NotificationRenderManager notificationRenderManager;

    public SpaceHomePageEventListener(UserAccessor userAccessor,
        NotificationManager notificationManager, DataSourceFactory dataSourceFactory, I18NBeanFactory i18NBeanFactory,
        LocaleManager localeManager, MultiQueueTaskManager taskManager,
        NotificationRenderManager notificationRenderManager)
    {
        this.userAccessor = userAccessor;
        this.notificationManager = notificationManager;
        this.dataSourceFactory = dataSourceFactory;
        this.i18NBeanFactory = i18NBeanFactory;
        this.localeManager = localeManager;
        this.taskManager = taskManager;
        this.notificationRenderManager = notificationRenderManager;
    }

    @EventListener
    public void onSpaceHomePageCreate(SpaceBlueprintHomePageCreateEvent event) {
        if (!TeamSpaceConstants.TEAM_SPACE_COMPLETE_KEY.equals(event.getSpaceBlueprint().getModuleCompleteKey())) {
            return;
        }
        String members = (String) event.getContext().get("members");
        String description = (String) event.getContext().get("description");

        final Space space = event.getSpace();
        String[] usernames = members.split(",");
        List<ConfluenceUser> users = Lists.newArrayList();
        for (String username: usernames)
        {
            final ConfluenceUser user = userAccessor.getUserByName(username);
            if (user != null)
            {
                notificationManager.addSpaceNotification(user, space);
                users.add(user);
            }
        }

        ConfluenceUser creator = event.getCreator();

        if (NotificationTemplate.ADG.isEnabled("team.space"))
        {
            sendNotifications(creator, description, space, users);
        }
    }

    private void sendNotifications(ConfluenceUser creator, String description, Space space, List<ConfluenceUser> users)
    {
        List<PairType> allTeamMembers = Lists.newArrayList();
        HashMap<String, DataHandler> allAvatarHandlers = Maps.newHashMap();
        Map<ConfluenceUser, String> avatarUserMap = Maps.newHashMap();

        // Add sender Avatar
        DataHandler avatarHandler = new DataHandler((dataSourceFactory.getAvatar(creator)));
        allAvatarHandlers.put(avatarHandler.getName(), avatarHandler);
        avatarUserMap.put(creator, avatarHandler.getName());

        for (ConfluenceUser user : users)
        {
            avatarHandler = new DataHandler((dataSourceFactory.getAvatar(user)));
            allAvatarHandlers.put(avatarHandler.getName(), avatarHandler);
            avatarUserMap.put(user, avatarHandler.getName());
            allTeamMembers.add(new PairType(user, avatarHandler.getName()));
        }

        DataHandler senderAvatarDataHandler = new DataHandler(dataSourceFactory.getAvatar(creator));

        NotificationContext context = new NotificationContext();
        context.put("spaceName", space.getName());
        context.put("spaceUrl", space.getUrlPath());
        context.put("teamSpaceType", TEAM_SPACE_TYPE);
        context.put("sender", creator);
        context.put("contentSummary", description);
        context.put("avatarCid", senderAvatarDataHandler.getName());

        context.put("spaceKey", space.getKey());
        context.put("homePageUrl", space.getHomePage().getUrlPath());
        context.put("actionLinks", getWebItemLinks("email.adg.space.action.links", context));
        notificationRenderManager.attachActionIconImages("email.adg.space.action.links", context);

        populateIcons(context);

        for (ConfluenceUser user : users)
        {
            if (user.equals(creator)) // don't send notification to the space creator
            {
                continue;
            }

            List<PairType> teamMembers = Lists.newArrayList(allTeamMembers);
            teamMembers.remove(new PairType(user, avatarUserMap.get(user))); // exclude email recipient from the list
            context.put("userList", teamMembers);

            HashMap<String, DataHandler> teamAvatarHandlers = Maps.newHashMap(allAvatarHandlers);
            teamAvatarHandlers.remove(avatarUserMap.get(user));
            sendTeamSpaceNotification(creator, user, convertToDataSource(teamAvatarHandlers), context);
        }

    }

    private List<SimpleLink> getWebItemLinks(String section, NotificationContext context)
    {
        List<WebItemModuleDescriptor> webItems = notificationRenderManager.getDisplayableItems(section, context);

        List<SimpleLink> emailLinks = Lists.newArrayList();
        for (WebItemModuleDescriptor webItem : webItems)
        {
            String url = webItem.getLink().getRenderedUrl(context.getMap());
            String i18nKey = webItem.getWebLabel().getKey();
            emailLinks.add(new SimpleLink(i18nKey, url));
        }
        return emailLinks;
    }

    private void sendTeamSpaceNotification(ConfluenceUser creator, ConfluenceUser receivingUser, List<DataSource> teamMemberAvatarHandlers, NotificationContext context)
    {
        I18NBean i18n = i18NBeanFactory.getI18NBean(localeManager.getLocale(receivingUser));
        String subject = i18n.getText("confluence.blueprints.space.team.email.subject", new String[]{creator.getFullName()});
        taskManager.addTask("mail", createNotificationTask(creator, receivingUser, subject, context, teamMemberAvatarHandlers));
    }

    private MailQueueItem createNotificationTask(ConfluenceUser creator, ConfluenceUser receiver, String subject,
        NotificationContext context, List<DataSource> avatarHandlers)
    {
        PreRenderedMailNotificationQueueItem.Builder builder = PreRenderedMailNotificationQueueItem.with(receiver, TEAM_SPACE_EMAIL_SOY_TEMPLATE, subject)
                                                                 .andSender(creator)
                                                                 .andTemplateLocation(TEAM_SPACE_EMAIL_COMPLETE_KEY)
                                                                 .andContext(context.getMap())
                                                                 .andRelatedBodyParts(avatarHandlers)
                                                                 .andRelatedBodyParts(iconResources)
                                                                 .andRelatedBodyParts(imagesUsedByChromeTemplate().get());
        return builder.render();
    }

    private void populateIcons(NotificationContext context)
    {
        iconResources.add(dataSourceFactory.forPlugin(TeamSpaceConstants.TEAM_SPACE_MODULE_KEY).get()
            .resourceFromModuleByName(TEAM_SPACE_EMAIL_RESOURCES, TEAM_SPACE_ICON_KEY).get()); // team space icon
        iconResources.addAll(context.getTemplateImageDataSources());
    }

    private List<DataSource> convertToDataSource(Map<String, DataHandler> avatarDataHandlers)
    {
        return Lists.newArrayList(Iterables.transform(avatarDataHandlers.values(),
            new Function<DataHandler, DataSource>()
            {
                public DataSource apply(DataHandler input)
                {
                    return input.getDataSource();
                }
            }));
    }

    private Maybe<Iterable<DataSource>> imagesUsedByChromeTemplate()
    {
        return dataSourceFactory.forPlugin(EMAIL_RESOURCE_KEY).get()
                                .resourcesFromModules("chrome-template", PluginDataSourceFactory.FilterByType.IMAGE);
    }
}
