package com.atlassian.confluence.plugins.retrospectives;

import com.atlassian.confluence.plugins.SoftwareBlueprintsContextProviderHelper;
import com.atlassian.confluence.plugins.common.event.SoftwareBPAnalyticEventUtils;
import com.atlassian.confluence.plugins.createcontent.api.contextproviders.AbstractBlueprintContextProvider;
import com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContext;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @since 2.1
 */
public class RetrospectivesContextProvider extends AbstractBlueprintContextProvider {
    public static final String TEMPLATE_PROVIDER_PLUGIN_KEY =
            "com.atlassian.confluence.plugins.confluence-software-blueprints:retrospective-resources";

    private static final String USERS_TEMPLATE = "Confluence.Blueprints.Common.users.soy";
    private static final String PARTICIPANTS_KEY = "userKeys";

    // legacy soy templates used by older versions of the Retrospectives blueprint
    private static final String USER_LIST_TEMPLATE = "Confluence.Blueprints.Common.userList.soy";
    private static final String USER_TABLE_TEMPLATE = "Confluence.Blueprints.Common.userTable.soy";

    private final SoftwareBlueprintsContextProviderHelper helper;

    public RetrospectivesContextProvider(SoftwareBlueprintsContextProviderHelper helper) {
        this.helper = helper;
    }

    @Override
    protected BlueprintContext updateBlueprintContext(BlueprintContext context) {
        Map<String, Object> soyContext = new HashMap<>();

        String participantParam = (String) context.get("retro-participants");
        if (isNotBlank(participantParam)) {
            soyContext.put(PARTICIPANTS_KEY, participantParam.split(","));
            context.put("participants", helper.renderFromSoy(TEMPLATE_PROVIDER_PLUGIN_KEY, USERS_TEMPLATE, soyContext));
        }

        context.setTitle((String) context.get("retro-title"));
        context.put("currentDateLozenge", helper.createStorageFormatForToday());

        addLegacyTemplateVariables(context, soyContext);
        doAnalytic(participantParam);

        return context;
    }

    // can't really get rid of these legacy variables without potentially breaking old custom space blueprints :(
    private void addLegacyTemplateVariables(BlueprintContext context, Map<String, Object> soyContext) {
        context.put("currentDate", helper.serverFormatDate(new Date()));
        if (soyContext.containsKey(PARTICIPANTS_KEY)) {
            context.put("participantList",
                    helper.renderFromSoy(TEMPLATE_PROVIDER_PLUGIN_KEY, USER_LIST_TEMPLATE, soyContext));
            context.put("participantTable",
                    helper.renderFromSoy(TEMPLATE_PROVIDER_PLUGIN_KEY, USER_TABLE_TEMPLATE, soyContext));
        }
    }

    private void doAnalytic(String participants) {
        ConfluenceUser authenticatedUser = AuthenticatedUserThreadLocal.get();
        if (authenticatedUser != null && authenticatedUser.getKey().getStringValue().equals(participants)) {
            helper.publishAnalyticEvent(SoftwareBPAnalyticEventUtils.RETROSPECTIVE_UPDATE_PARTICIPANTS_EVENT_NAME);
        }
        helper.publishAnalyticEvent(SoftwareBPAnalyticEventUtils.RETROSPECTIVE_CREATE_EVENT_NAME);
    }
}
