package com.atlassian.confluence.plugins.jirareports;

import com.atlassian.applinks.api.CredentialsRequiredException;
import com.atlassian.applinks.api.ReadOnlyApplicationLink;
import com.atlassian.confluence.extra.jira.Channel;
import com.atlassian.confluence.extra.jira.JiraIssuesManager;
import com.atlassian.confluence.renderer.radeox.macros.MacroUtils;
import com.atlassian.confluence.util.velocity.VelocityUtils;
import org.apache.log4j.Logger;
import org.jdom.Element;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.replace;

/**
 * This class contains service relate to jira
 *
 * @since 2.1
 */
public class JiraIssuesHelper {

    private JiraIssuesManager jiraIssuesManager;
    private static final Logger LOGGER = Logger.getLogger(JiraIssuesHelper.class);
    private static final Set<String> DEFAULT_COLUMNS = Stream.of("key", "summary", "status", "type").collect(toSet());
    private static final String XML_SEARCH_REQUEST_URI = "/sr/jira.issueviews:searchrequest-xml/temp/SearchRequest.xml?jqlQuery=";
    private static final String JIRA_ISSUES_TEMPLATE = "com/atlassian/confluence/plugins/jirareports/velocity/jira-issues.html.vm";
    private static final List<String> DEFAULT_ISSUE_TYPES = Arrays.asList("Epic", "New Feature", "Improvement", "Bug");
    private static final String EMPTY_RESULT = "";
    private static final int DEFAULT_TOTAL_ISSUES = 0;
    private static final String DEFAULT_OPTION_VALUE = "-1";

    public JiraIssuesHelper(JiraIssuesManager jiraIssuesManager) {
        this.jiraIssuesManager = jiraIssuesManager;
    }

    /**
     * The function renders jira issues
     *
     * @param channel jira issue channel
     * @return render html
     */
    public String renderJiraIssues(Channel channel) {
        if (channel != null) {
            Element jiraIssuesElement = channel.getChannelElement();
            List<Element> items = jiraIssuesElement.getChildren("item");
            if (items != null && !items.isEmpty()) {
                StringBuilder jiraIssues = new StringBuilder();
                Map<String, List<Element>> mapIssueTypes = getMapIssueType(items);
                appendDefaultIssueTypes(mapIssueTypes, jiraIssues);
                appendOtherIssueTypes(mapIssueTypes, jiraIssues);
                return jiraIssues.toString();
            }
        }
        return EMPTY_RESULT;
    }

    /**
     * The function get total issues number
     *
     * @param channel jira issue channel
     * @return total issues number
     */
    public int getTotalIssueNumber(Channel channel) {
        if (channel != null) {
            Element jiraIssuesElement = channel.getChannelElement();
            Element totalItemsElement = jiraIssuesElement.getChild("issue");
            if (totalItemsElement != null) {
                return Integer.parseInt(totalItemsElement.getAttributeValue("total"));
            }
        }
        return DEFAULT_TOTAL_ISSUES;
    }

    /**
     * The function renders html for default issue types
     *
     * @param mapIssueTypes map issue type
     * @param jiraIssues    jiraIssues html
     */
    private void appendDefaultIssueTypes(Map<String, List<Element>> mapIssueTypes, StringBuilder jiraIssues) {
        for (String issueType : DEFAULT_ISSUE_TYPES) {
            if (mapIssueTypes.get(issueType) != null) {
                jiraIssues.append(velocityRender(mapIssueTypes, issueType));
            }
        }
    }

    /**
     * The function renders html for not default issue types
     *
     * @param mapIssueTypes map issue type
     * @param jiraIssues    jiraIssues html
     */
    private void appendOtherIssueTypes(Map<String, List<Element>> mapIssueTypes, StringBuilder jiraIssues) {
        Set<String> keys = mapIssueTypes.keySet();
        for (String key : keys) {
            if (!DEFAULT_ISSUE_TYPES.contains(key)) {
                jiraIssues.append(velocityRender(mapIssueTypes, key));
            }
        }
    }

    /**
     * Using velocity to render html
     *
     * @param mapIssueTypes map issue type
     * @param issueType     issue type
     * @return html data
     */
    private String velocityRender(Map<String, List<Element>> mapIssueTypes, String issueType) {
        Map<String, Object> context = MacroUtils.defaultVelocityContext();
        context.put("title", issueType);
        context.put("issues", mapIssueTypes.get(issueType));
        return VelocityUtils.getRenderedTemplate(JIRA_ISSUES_TEMPLATE, context);
    }

    /**
     * The function separate list issues by issue type
     *
     * @param elements jira issues xml
     * @return map issue type
     */
    private Map<String, List<Element>> getMapIssueType(List<Element> elements) {
        Map<String, List<Element>> mapIssueType = new HashMap<>();
        String type;
        for (Element element : elements) {
            type = element.getChild("type").getValue();
            if (mapIssueType.get(type) == null) {
                List<Element> list = new ArrayList<>();
                list.add(element);
                mapIssueType.put(type, list);
            } else {
                mapIssueType.get(type).add(element);
            }
        }
        return mapIssueType;
    }

    /**
     * The function build project version jql
     *
     * @param contextMap request map
     * @return project version jql
     */
    public String buildProjectVersionJQL(Map<String, Object> contextMap) {
        String project = (String) contextMap.get("jira-reports-project");
        if (project == null || project.equals(DEFAULT_OPTION_VALUE)) {
            return null;
        }

        StringBuilder url = new StringBuilder();
        url.append("project=\"");
        url.append(replace(project, "'", "''"));
        url.append("\"");

        String versions = (String) contextMap.get("multiVersion");
        if (!isBlank(versions)) {
            url.append(" AND fixVersion in (");
            url.append(replace(versions, "'", "''"));
            url.append(")");
        }
        return url.toString();
    }

    /**
     * Get xml channel impersonally, always return fresh data.
     *
     * @param jqlQuery jql pass to jira server
     * @param appLink  jira server connection
     * @return Channel
     */
    public Channel getChannel(ReadOnlyApplicationLink appLink, String jqlQuery, int maxResult) {
        String hostName = appLink.getRpcUrl().toString();
        String requestJiraUrl = hostName + XML_SEARCH_REQUEST_URI + jqlQuery + "&tempMax=" + maxResult;
        try {
            // fix for CONFDEV-20458, by changing cache parameter to false
            return jiraIssuesManager.retrieveXMLAsChannel(requestJiraUrl, DEFAULT_COLUMNS, appLink, false, false);
        } catch (CredentialsRequiredException e) {
            return getChannelByAnonymous(requestJiraUrl, appLink);
        } catch (Exception e) {
            LOGGER.error("Can not retrieve jira issues", e);
        }
        return null;
    }

    /**
     * The service get channel by anonymous user
     *
     * @param url     url pass to jira server
     * @param appLink jira server connection
     * @return Channel
     */
    private Channel getChannelByAnonymous(String url, ReadOnlyApplicationLink appLink) {
        try {
            return jiraIssuesManager.retrieveXMLAsChannelByAnonymous(url, DEFAULT_COLUMNS, appLink, false, true);
        } catch (Exception e) {
            LOGGER.error("Can not retrieve jira issues", e);
        }
        return null;
    }
}
