# MediaViewer Integration Guide

Integrating MediaViewer is a matter of adding a couple of script tags and a few lines of JavaScript to your page.  
This guide will first walk you through the initial setup and then mention a few other topics.

- [Setup](#setup)
- [Enabling Plugins](#enabling-plugins)
- [Disabling Viewers](#disabling-viewers)
- [Video Player](#video-player)
- [Translations](#translations)
- [Generating Previews](#generating-previews)

## Setup

It's 5 simple steps to set everything up.

### 1. Check out MediaViewer

You can install MediaViewer via npm:

    npm install @atlassian/mediaviewer@2.0.0

### 2. Add Dependencies

First of all, make sure that you have the following libraries on your page:

    <link rel="stylesheet" href="node_modules/aui-dist/aui/css/aui.css"/>

(we may get rid of this dependency eventually)

Other dependencies:

- jquery
- backbone
- underscore

*Note*: Most versions of `AUI` are supported. However, make sure that you always include `AUI` before `backbone`.

### 3. Add MediaViewer

Add 3 files to your page and you're ready to go:

    <script src="node_modules/atlassian-mediaviewer/dist/mediaviewer.all.js"></script>

Webpack will dynamically load all the needed chunks, so make sure your product can do that.

### 4. Create an Instance of MediaViewer

To create an instance of MediaViewer, all you have to do is the following:

    var viewer = new MediaViewer({
        // enable MiniMode plugin
        enableMiniMode: true,
        assets: {
           basePath: 'dist/lib/'
        }
    });

There's other configuration options as well, check out the source code of `mediaviewer.js` for now.

### 5. Show a File

First, tell MediaViewer about the files it should be aware of:

    viewer.setFiles([
        {
            id: 0,
            src: 'image.png',
            type: 'image/png',
            title: 'image'
        },
        {
            id: 1,
            src: 'document.pdf',
            type: 'application/pdf',
            title: 'document'
        }
    ]);

Now open the viewer and show a file:

    viewer.open();
    viewer.showFileWithQuery({id: 0});

et voilà - you are now showing the file in your browser.

## Enabling Plugins

As soon as plugins are included on the page, they register themselves with MediaViewer. However, some of them need to be activated with certain configuration first. Here's an example:

    // enable MiniMode plugin
    var viewer = new MediaViewer({ enableMiniMode: true });

## Disabling Viewers

By default, all viewers are enabled - even though might not yet be included on the page. To completely disable viewers, you need to overwrite the default configuration and mention all the viewers that you want to use instead:

    // disable document viewer
    var viewer = new MediaViewer({
      viewers: ['image', 'video']
    });

Just make sure that the enabled viewers are either on the page or that you setup lazy loading for the viewers.

## Video Player

### Enabling HD Videos

In addition to setting `file.src`, also set `file.src_hd`. If this property is set, the HD toggle button
shows up automatically

### Enabling Youtube Videos

Set the MIME-type of the file to `video/youtube` and add the Video.js youtube-plugin to the page.

### Specifying a Separate Download Url

If you want to specify a separate url for downloading a file via the download button, simply specify the
`srcDownload` in addtion to `src` on MediaViewer.

## Translations

Use the `i18n` config property:

    var viewer = new MediaViewer({
      enableMiniMode: true,
      enableListLoop: true,
      assets: {
        basePath: 'dist/lib/'
      },
      i18n: AJS.I18n.keys,
      isPreviewGenerated: isPreviewGenerated,
      generatePreview: generatePreview,
      watermark: {
        imgSrc: 'http://dt-fileviewer-example-assets.s3.amazonaws.com/watermark-example.png',
        linkUrl: 'https://www.atlassian.com'
      }
    });


Just check the examples.

## Fetching Tokens

In case your loading files directly from the Media API you'll need to fetch a specific token
for every file first. Media Viewer allows you to provide a handler for exactly that use case.

Here's an example:

    var viewer = new MediaViewer({
        fetchToken: fetchTokenHandler
    });

The handler will be invoked with the original file object and is expected to return a promise resolving with
an overwrites object. In particular, if you want to modify the `src` attribute of the file, you have to return
an object with nothing but the updated `src` (e.g. `{ src: updatedSrc, srcDownload: updatedSrcDownload }`).

Here's an example handler that fetches a token from some endpoint and updated the file src:

    function fetchTokenHandler(originalFile) {
        var endpoint = '/token/file/' + originalFile.get('id');
        return $.get(endpoint).pipe(function (response) {
            var updatedSrc = originalFile.get('src') + '?token=' + response.data.token;
            var updatedSrcDownload = originalFile.get('srcDownload') + '?token=' + response.data.token;
            return {
                src: updatedSrc,
                srcDownload: updatedSrcDownload
            };
        });
    }


## Generating Previews

When MediaViewer can't preview a file, it can ask a different service to provide it
with a converted, viewable version of the file.

To do that, you have to configure a custom conversion service like this:

1. Make sure that MediaViewer doesn't support the given MIME-type natively.

        new File({ type: 'not/supported' });

    **Note: MediaViewer will only call the conversion service, if it doesn't support the given file type. Here's a [list of supported types](https://extranet.atlassian.com/display/FIL/Supported+filetypes?src=contextnavpagetreemode).**

2. Provide the two functions `isPreviewGenerated()` and `generatePreview()` in the configuration.

        var viewer = new MediaViewer({
            isPreviewGenerated: isPreviewGenerated,
            generatePreview: generatePreview,
            // ....
        });

3. Implement `isPreviewGenerated()`:

        // This function gets the file model passed in and
        // has to return a promise (you can use jQuery promises).
        var isPreviewGenerated = function (file) {

            // if there is a preview, resolve with:
            // true, url of the converted file and
            // type of the converted file and optionally
            // additional overwrites for the converted file
            var overwrites = { src_hd: 'urlToHdVersion' };
            return $.when(true, urlOfConvertedFile, typeOfConvertedFile, overwrites);

            // if there is no converted file yet, simply resolve with false
            // to trigger the conversion
            return $.when(false);

            // if there is a failure, reject the promise
            return $.reject(new Error('conversion service not available'));
        };

4. Implement `generatePreview()`:

        // This function, too, gets the file model passed in and
        // has to return a promise (you can use jQuery promises).
        var generatePreview = function (file) {

            // once you're done, resolve with
            // the url of the converted file and
            // the type of the converted file and optionally
            // additional overwrites for the converted file
            var overwrites = { src_hd: 'urlToHdVersion' };
            return $.when(urlOfConvertedFile, typeOfConvertedFile, overwrites);

            // if there is a failure, reject the promise
            return $.reject(new Error('conversion service not available'));
        };

When `isPreviewGenerated()` returns `false`, a spinner will show up and wait for
`generatePreview()` to resolve. Afterwards, the converted file will be shown.
