define("cp/service/comments-service",
    [
        "underscore",
        "ajs",
        "jquery"
    ],
    function(
        _,
        AJS,
        $
        ) {
        'use strict';

        function CommentsService(fileID, version) {
            this.url = AJS.contextPath() + "/rest/files/1.0/files/" +  fileID + "/comments";
            this.version = version;
        }

        var _makePropRetriever = function (obj, defVal) {
            return function(path) {
                return AJS.defaultIfUndefined(path, {rootObject: obj, defaultValue: defVal});
            };
        };

        var _toComment = function (response) {
            var getOrEmpty = _makePropRetriever(response, '');
            var getOrUndefined = _makePropRetriever(response, undefined);
            var createDate = getOrUndefined('history.createdDate') || new Date();

            return {
                id: getOrEmpty('id'),
                author: {
                    name: getOrEmpty('history.createdBy.displayName'),
                    avatar: getOrEmpty('history.createdBy.profilePicture.path'),
                    profile: AJS.contextPath() + "/display/~" + getOrEmpty('history.createdBy.username')
                },
                comment: getOrEmpty('body.view.value'),
                editorFormat: getOrEmpty('body.editor.value'),
                date: createDate,
                hasDeletePermission: response.hasDeletePermission,
                hasEditPermission: response.hasEditPermission,
                hasReplyPermission: response.hasReplyPermission,
                hasResolvePermission: response.hasResolvePermission
            };
        };

        var _toAnnotation = function (response) {
            var getOrUndefined = _makePropRetriever(response, undefined);
            var getOrFalse = _makePropRetriever(response, false);

            return _.extend({
                pageNumber: getOrUndefined('anchor.page'),
                position: [getOrUndefined('anchor.x'), getOrUndefined('anchor.y')],
                resolved: getOrFalse('resolved.value')
            }, _toComment(response));
        };

        var _fromComment = function (comment, isUpdate) {
            var changedAttributes = comment.changedAttributes();

            if (changedAttributes.hasOwnProperty("resolved") && isUpdate) {
                return {resolved: comment.get("resolved")};
            }

            var annotation = {
                parentId: comment.get("parentId"),
                commentBody: comment.get("editorFormat")
            };

           // @todo: comments-service shouldn't need to be aware of pageNumber
            if (comment.get("position") || comment.get("pageNumber")) {
                annotation = _.extend({}, annotation, {
                    anchor: {
                        x: comment.get("position")[0],
                        y: comment.get("position")[1],
                        page: comment.get("pageNumber"),
                        type: "pin"
                    }
                });
            }

            return annotation;

        };

        CommentsService.prototype._makeUrl = function(path) {
            if (!path) {
                path = "";
            }
            return this.url + path + (this.version ? ("?attachmentVersion=" + this.version) : "");
        };

        CommentsService.prototype.getAnnotations = function(params) {
            var dfd = $.Deferred();
            var opts = _.extend({}, params);
            $.ajax({
                url: this._makeUrl(),
                type: "GET",
                data: {
                    'limit': opts.limit,
                    'start': opts.start
                },
                dataType: "json"
            }).then(function(data) {
                var annotations = _.map(data.results, _toAnnotation);
                dfd.resolve(annotations);
            }).fail(function (jqXHR, textStatus, errorThrown) {
                dfd.reject(jqXHR, textStatus, errorThrown);
            });
            return dfd.promise();
        };

        CommentsService.prototype.getReplies = function(params) {
            var dfd = $.Deferred();
            var opts = _.extend({}, params);
            $.ajax({
                url: this._makeUrl("/" + params.parentId),
                type: "GET",
                data: {
                    'limit': opts.limit,
                    'start': opts.start
                },
                dataType: "json"
            }).then(function(data) {
                var getOrArray = _makePropRetriever(data, []);
                var response = getOrArray("children");
                var replies = _.map(response, _toComment);
                dfd.resolve(replies);
            }).fail(function (jqXHR, textStatus, errorThrown) {
                dfd.reject(jqXHR, textStatus, errorThrown);
            });
            return dfd.promise();
        };

        CommentsService.prototype.save = function(comment) {
            var dfd = $.Deferred();
            var method = comment.get("id") ? "PUT" : "POST";
            var url = (method === "PUT") ? this._makeUrl("/" + comment.get("id")) : this._makeUrl();

            $.ajax({
                url: url,
                type: method,
                data: JSON.stringify(_fromComment(comment, (method === "PUT"))),
                dataType: "json",
                contentType: "application/json; charset=utf-8"
            }).then(function (data) {
                dfd.resolve(_toComment(data));
            }).fail(function (jqXHR, textStatus, errorThrown) {
                dfd.reject(jqXHR, textStatus, errorThrown);
            });

            return dfd.promise();
        };

        CommentsService.prototype.remove = function(comment) {
            var dfd = $.Deferred();
            $.ajax({
                url: this._makeUrl("/" + comment.get("id")),
                type: "DELETE",
                dataType: "json"
            }).then(function () {
                dfd.resolve();
            }).fail(function (jqXHR, textStatus, errorThrown) {
                dfd.reject(jqXHR, textStatus, errorThrown);
            });

            return dfd.promise();
        };

        return CommentsService;

    }
);
