define("cp/component/versions/version-navigation-dialog-view",
    [
        "ajs",
        "backbone",
        "jquery",
        "underscore",
        "MediaViewer",
        "cp/component/versions/file-version",
        "cp/component/versions/versions",
        "core/template-store-singleton"
    ],
    function (
        AJS,
        Backbone,
        $,
        _,
        FileViewer,
        Version,
        Versions,
        templateStore) {

        "use strict";

        /**
         * Version navigation dialog
         */
        var VersionNavigationDialogView = Backbone.View.extend({

            dialogId: "cp-dialog-version",
            events: {
                "click .versionFiles li:not(.current)": "_onClickFileVersion"
            },

            initialize: function (options) {
                if (options) {
                    this._mediaViewer = options.mediaViewer;
                    this.$btnTrigger = options.$btnTrigger;
                    this._file = this._mediaViewer.getCurrentFile();
                }

                this.dialog = AJS.InlineDialog(
                    this.$btnTrigger.find("span.cp-file-icon"),
                    this.dialogId,
                    function (content, trigger, showPopup) {
                        // moving $el would remove delegated events on $el; avoid by detaching first
                        content.append(this.$el.detach());
                        this.render();
                        showPopup();
                    }.bind(this),
                    {
                        hideDelay: null,
                        offsetY: 0,
                        noBind: true,
                        hideCallback: function() {
                            this.$btnTrigger.removeClass("active");
                        }.bind(this)
                    }
                );

                this._mediaViewer.once("fv.close", this.teardown.bind(this));

                this.$btnTrigger.on("click.versionnav", function() {
                    if (this.$el.is(":visible")) {
                        this.dialog.hide();

                        AJS.trigger("analyticsEvent", {
                            name: "confluence-spaces.previews.versions.click-title-to-close"
                        });
                    } else {
                        this.dialog.show();
                        this.$btnTrigger.addClass("active");
                    }
                }.bind(this));
            },

            teardown: function () {
                this.$btnTrigger.off("click.versionnav");
                this.dialog.hide();
                // InlineDialog doesn't have a method to destroy itself
                $("#inline-dialog-cp-dialog-version").remove();
            },

            render: function () {
                this.$el.html(templateStore.get("Versions.versionNavigationDialog")());

                this.$(".spinner-wrap").spin("medium", {top: 0, left: 0});

                var latestVersion = this._file.getLatestVersion();
                this._file.get('versions').fetchVersions(true)
                    .done(function (versions) {
                            if (+versions[0].version > +latestVersion.get("version")) {
                                this._file.getLatestVersion().set("stale", true);
                                this._mediaViewer.getView().fileContentView.getLayerForName('version-message').render();
                                versions = _.filter(versions, function(version) {
                                    return version.version <= latestVersion.get("version");
                                });
                            }

                            this.$(".versions-container").html(templateStore.get("Versions.versionHistory")({
                                versions: new Backbone.Collection(versions).toJSON(),
                                selectedVersion: this._mediaViewer.getCurrentFile().get("version")
                            }));
                        }.bind(this))
                    .fail(function() {
                            this.$(".versions-container").html(AJS.messages.warning({
                                title: AJS.I18n.getText("cp.version.versions-list.error.header"),
                                body: AJS.I18n.getText("cp.version.versions-list.error.message"),
                                closeable: false
                            }));
                        }.bind(this));

                AJS.trigger("analyticsEvent", {
                    name: "confluence-spaces.previews.versions.open",
                    data: {fileType: this._file.get("type")}
                });

                return this;
            },

            _onClickFileVersion: function(e) {
                e.preventDefault();

                var version = this._file.get('versions').get($(e.currentTarget).attr("data-cid"));
                version.getFileVersion().done(function(selectedFile) {
                    this._mediaViewer.showFile(selectedFile);
                }.bind(this));

                var selectedVersion = version.get("version");
                var latestVersion = this._file.getLatestVersion().get("version");

                this._triggerClickFileVersionAnalyticsEvent(selectedVersion, latestVersion);
            },

            _triggerClickFileVersionAnalyticsEvent: function(selectedVersion, latestVersion) {
                var eventName = "confluence-spaces.previews.versions.view-previous";
                var versionDiff = latestVersion - selectedVersion;

                if (versionDiff >= 4) {
                    eventName += "4-and-older";
                } else if (versionDiff > 0) {
                    eventName += versionDiff;
                }

                if (versionDiff > 0) {
                    AJS.trigger("analyticsEvent", {
                        name: eventName
                    });
                }
            }
        });

        return VersionNavigationDialogView;
    });