define("cp/component/annotation/content-view",
    [
        "MediaViewer",
        "backbone",
        "underscore",
        "jquery"
    ],
function(
        FileViewer,
        Backbone,
        _,
        $
) {
    'use strict';

    // Amount of time to wait before hiding the controls when the mouse stops moving.
    var HIDE_CONTROLS_TIMEOUT = 500;

    // Amount of time between checking if the mouse is still moving.
    // Should be smaller than HIDE_CONTROLS_TIMEOUT to prevent flickering.
    var THROTTLE_MOUSEMOVE = HIDE_CONTROLS_TIMEOUT - 100;

    var KEYS = {
        PLUS: 187,
        MINUS: 189,
        PLUS_NUMPAD: 107,
        MINUS_NUMPAD: 109,
        PLUS_FF: 61,
        MINUS_FF: 173
    };

    var BaseView = FileViewer.require('core/base_viewer');
    var extensions = {
        annotationEvents: {
            'mousemove.contentView.show': '_showControlsOnMove',
            'mousemove.contentView.hide': '_hideControlsOnMove'
        },

        initialize: function() {
            this.events = _.extend(this.events || {}, this.annotationEvents);
            this._toggleControlsTimeout = null;
            this._autoToggleControls = true;
            $(document).on('keydown.viewerZoom', this._handleKeyboardZoom.bind(this));
        },

        // Show / hide controls based on mouse movements:
        //
        // - Show the controls when the mouse is moving over the content view.
        // - Hide the controls after a short delay when the mouse stops moving.
        // - Keep the controls open if the mouse is hovering over them.
        //
        // Only works if the child provides getControlsElement().

        _showControlsOnMove : _.throttle(function () {
            if (!this._autoToggleControls) { return; }
            this.showControls();
        }, THROTTLE_MOUSEMOVE),

        _hideControlsOnMove : _.throttle(function () {
            if (!this._autoToggleControls) { return; }
            this.hideControls();
        }, THROTTLE_MOUSEMOVE),


        showControls: function () {

            if (!this.getControlsElement) { return; }

            clearTimeout(this._toggleControlsTimeout);
            this.getControlsElement().show();

        },

        hideControls: function () {
            if (!this.getControlsElement) { return; }
            this._toggleControlsTimeout = this._setHideTimer();
        },

        _setHideTimer: function () {
            return setTimeout(function () {
                if (this.getControlsElement().is(':hover')) { return; }
                this.getControlsElement().fadeOut();
            }.bind(this), HIDE_CONTROLS_TIMEOUT);
        },

        autoToggleControls: function (toggle) {
            this._autoToggleControls = toggle;
            if (!this._autoToggleControls) {
                clearTimeout(this._toggleControlsTimeout);
            }
        },

        _handleKeyboardZoom: function (e) {
            var metaKey = (e.ctrlKey || e.metaKey);

            if (this.zoomIn && metaKey &&
                (e.which === KEYS.PLUS || e.which === KEYS.PLUS_NUMPAD || e.which === KEYS.PLUS_FF)) {
                this.zoomIn();
                e.preventDefault();
            }

            if (this.zoomOut && metaKey &&
                (e.which === KEYS.MINUS || e.which === KEYS.MINUS_NUMPAD || e.which === KEYS.MINUS_FF)) {
                this.zoomOut();
                e.preventDefault();
            }
        }
    };

    Object.keys(extensions).forEach(function (key) {
        BaseView[key] = extensions[key];
    });

    return BaseView;
});