define('confluence/page-hierarchy/service/progress-runner', [
        'ajs',
        'confluence/page-hierarchy/util/execute-if-function'
    ],
    function (AJS,
              executeIfFunction) {
        var LONG_TASK_REST_ENDPOINT = '/rest/api/longtask/<taskId>';
        var REGULAR_PROGRESS_POLL_INTERVAL = 1000;
        var QUICK_PROGRESS_POLL_INTERVAL = 172;
        var PROGRESS_INDICATOR_SELECTOR = '#progress-indicator';
        var STATUS_SELECTOR = '#status';
        var WARNING_TYPE = 'WARN';
        var $ = AJS.$;

        /**
         * A Progress Runner service that queries the Long Running Task API for the status of
         * a copy or delete operation.
         * @param {object} options Options. See usage for which options are available.
         * @constructor
         */
        var Runner = function (options) {
            this.state = options.state;
            this.$dialog = options.dialog;
            this.onStatus = options.onStatus;
            this.defaultMessages = options.defaultMessages;
            this.onProgress = options.onProgress;
            this.onComplete = options.onComplete;
            this.ERROR_UNKNOWN = options.ERROR_UNKNOWN;
        };

        /**
         * Updates the status of the progress dialog
         * @param {{key:string,args:Array}} status Status reported by LongRunningTask
         * @returns {undefined}
         * @private
         */
        Runner.prototype.updateDisplayStatus = function (status) {
            var $status = this.$dialog.find(STATUS_SELECTOR);
            $status.html(AJS.escapeEntities(this.onStatus(status)));
        };

        /**
         * Main method that polls for progress updates.
         * @returns {undefined}
         */
        Runner.prototype.run = function () {
            var that = this;
            var timeout;
            var timesPolled = 0;
            AJS.progressBars.update(PROGRESS_INDICATOR_SELECTOR, 0);
            var process = function () {
                $.ajax({
                    url: AJS.contextPath() + LONG_TASK_REST_ENDPOINT.replace('<taskId>', that.state.getState().taskId)
                }).done(function (data) {
                    var messages = _getMessages(data, that.defaultMessages);
                    that.updateDisplayStatus(messages.status);
                    executeIfFunction(that.onProgress, messages);

                    if (!data.successful) {
                        clearTimeout(timeout);
                        return that.errorResult(messages);
                    }

                    AJS.progressBars.update(PROGRESS_INDICATOR_SELECTOR, data.percentageComplete / 100);

                    if (+data.percentageComplete >= 100) {
                        clearTimeout(timeout);
                        if (data.successful) {
                            that.successResult(messages);
                        } else {
                            that.errorResult(messages);
                        }

                        return;
                    }

                    updateStatusPoller();
                }).fail(function () {
                    clearTimeout(timeout);
                    that.errorResult();
                });
            };
            var updateStatusPoller = function () {
                timeout = setTimeout(process,
                    timesPolled++ < 5 ? QUICK_PROGRESS_POLL_INTERVAL : REGULAR_PROGRESS_POLL_INTERVAL);
            };
            process();
        };

        /**
         * Trigger the success result, and set any errors
         * @param {object} messages {@see #_getMessages}
         * @returns {undefined}
         * @private
         */
        Runner.prototype.successResult = function (messages) {
            AJS.progressBars.update(PROGRESS_INDICATOR_SELECTOR, 1);
            // Even a successful result could still contain some problems with some
            // of the pages but it is not a total error
            if (messages && messages.errors && messages.errors.length) {
                this.sortAndSetErrors(messages.errors);
            }

            executeIfFunction(this.onComplete);
        };

        /**
         * Sort the errors into warnings and errors and update the state
         * @param {Array} restErrors Array of errors/warnings coming from the server
         * @returns {undefined}
         * @private
         */
        Runner.prototype.sortAndSetErrors = function (restErrors) {
            var sorted = this.sortErrors(restErrors);
            this.state.setErrors(sorted.errors);
            this.state.setWarnings(sorted.warnings);
        };

        /**
         * Sort between errors and warnings
         * @param {Array} restErrors Array of errors/warnings coming from the server
         * @returns {{errors: Array, warnings: Array}} Object containing errors in one array, warnings in another
         * @private
         */
        Runner.prototype.sortErrors = function (restErrors) {
            var errors = [];
            var warnings = [];
            for (var i = 0; i < restErrors.length; i++) {
                var error = restErrors[i];
                if (error.type === WARNING_TYPE) {
                    warnings.push(error);
                } else {
                    errors.push(error);
                }
            }

            return {errors: errors, warnings: warnings};
        };

        /**
         * Trigger the error result, and set the errors
         * @param {object} messages {@see #_getMessages}
         * @returns {undefined}
         * @private
         */
        Runner.prototype.errorResult = function (messages) {
            var state = this.state;

            AJS.progressBars.setIndeterminate(PROGRESS_INDICATOR_SELECTOR);
            if (messages && messages.errors && messages.errors.length) {
                this.sortAndSetErrors(messages.errors);
            } else {
                state.setErrors([{key: this.ERROR_UNKNOWN}]);
            }

            executeIfFunction(this.onComplete);
        };

        /**
         * Parses the messages from the status field of the long running task if any exist
         * @param {object} data  LongRunningTaskResult
         * @param {function} defaultMessages Default messages if parsing fails
         * @returns {{status:object,errors:object}} Messages from the server
         * @private
         */
        function _getMessages(data, defaultMessages) {
            if (data.messages && data.messages.length) {
                try {
                    return JSON.parse(data.messages[data.messages.length - 1].translation);
                } catch (e) {
                    return defaultMessages();
                }
            }

            return defaultMessages();
        }

        return Runner;
    });