define('confluence/page-hierarchy/service/page-tree', [
    'ajs',
    'confluence/legacy',
    'confluence/page-hierarchy/fetch/fetch-children',
    'confluence/page-hierarchy/fetch/fetch-children-with-incoming-links',
    'confluence/page-hierarchy/util/analytics-event',
    'confluence/page-hierarchy/util/execute-if-function'
], function (AJS,
             Confluence,
             fetchChildren,
             fetchChildrenWithIncomingLinks,
             analyticsEvent,
             executeIfFunction) {
    /**
     * Page Tree service that can display a page tree in the preview for Copy and Delete Hierarchy
     * @param {object} options Options. See usage for which options are available.
     * @returns {undefined}
     * @constructor
     */
    var PageTree = function (options) {
        this.loadingIndicator = options.loadingIndicator;
        this.onExpand = options.onExpand;
        this.incomingLinks = options.incomingLinks;
        this.restrictEditPages = options.restrictEditPages;
        this.rootContentId = options.rootContentId;
    };

    /**
     * Main method that loads a page tree into the root $childrenElement
     * @param {number} pageId Id of the parent page
     * @param {jQuery} $childrenElement jQuery Dom Element housing the children for the parent page
     * @returns {undefined}
     */
    PageTree.prototype.load = function (pageId, $childrenElement) {
        this.getChildren(pageId, $childrenElement);
        $childrenElement.on('click.get-children', this.childrenClickHandler.bind(this));
    };

    /**
     * Fetch children from the server and display in the page tree
     * @param {number} pageId   The page Id of the page to get children for.
     * @param {jQuery} $childrenContainer    The jQuery container for the children
     * @returns {undefined}
     * @private
     */
    PageTree.prototype.getChildren = function (pageId, $childrenContainer) {
        var that = this;
        var fetch = that.incomingLinks
            ? fetchChildrenWithIncomingLinks
            : fetchChildren;
        that.loadingIndicator.loading();
        fetch(pageId, that.rootContentId).done(function (data) {
            var children = data.results;
            that.loadingIndicator.done();
            $childrenContainer.data('is-fetched', true);
            $childrenContainer.html(Confluence.Templates.PageHierarchy.children({
                hasNext: data.hasNext,
                incomingLinks: that.incomingLinks,
                restrictEditPages: that.restrictEditPages,
                children: children
            }));
            that.expandChildren($childrenContainer);
        });
    };

    /**
     * Expands children in the preview
     * @param {jQuery} $childrenContainer The jQuery objet containing the children.
     * @returns {undefined}
     * @private
     */
    PageTree.prototype.expandChildren = function ($childrenContainer) {
        $childrenContainer.slideDown('fast');
        $childrenContainer.closest('.child').addClass('is-expanded');
        executeIfFunction(this.onExpand, $childrenContainer);
        analyticsEvent.publish(analyticsEvent.PREVIEW_EXPAND_CHILDREN);
    };

    /**
     * Collapses Children in the preview
     * @param {jQuery} $childrenContainer The jQuery objet containing the children.
     * @returns {undefined}
     * @private
     */
    PageTree.prototype.collapseChildren = function ($childrenContainer) {
        $childrenContainer.slideUp('fast');
        $childrenContainer.closest('.child').removeClass('is-expanded');
        analyticsEvent.publish(analyticsEvent.PREVIEW_COLLAPSE_CHILDREN);
    };

    /**
     * Handles click event on the preview page tree to expand or collapse children
     * @param {object} event The DOM click event.
     * @returns {undefined}
     * @private
     */
    PageTree.prototype.childrenClickHandler = function (event) {
        var that = this;
        var $target = $(event.target);
        var $parent = $target.closest('.child');
        if ($target.hasClass('indicator') && $parent.length && $parent.hasClass('has-children')) {
            var $childrenContainer = $parent.children('.children').first();
            if ($parent.hasClass('is-expanded')) {
                return that.collapseChildren($childrenContainer);
            }
            if ($childrenContainer.data('is-fetched')) {
                return that.expandChildren($childrenContainer);
            }
            that.getChildren($parent.data('page-id'), $childrenContainer);
        }
    };

    return PageTree;
});