define('confluence/ob/modules/tutorial/editor/tour', [
    'jquery',
    'aui/inline-dialog2'
], function(
        $
) {
    'use strict';

    var tour = {};
    var _STEPS = ['pageTitle', 'dragDrop', 'save'];

    var _nextStep = null;
    var popupId = 'onboarding-editor-tour-popup';
    var dialogId = 'onboarding-editor-tour-dialog';

    var _analytics = null;
    var EVENTS = {
        POPUP_NEXT_BUTTON_CLICK: "next.button.click.",
        DRAGDROP_NEXT_BUTTON_CLICK: 'next.button.click.dragDrop'
    };

    var _interval = null;

    var onFinishedTour = function() {};

    function createPopup(data, onNextClicked) {
        var context = $.extend({id: popupId}, data);
        var $el = $(Confluence.Templates.OB.tourPopup({
            data: context
        }));
        $el.on('click', '.next', _.bind((onNextClicked || onDefaultNextButtonClicked), $el, data.stepName));
        return $el.appendTo('body');
    }

    function updatePopupTrigger(data) {
        var $trigger = $(data.trigger);

        // setting trigger
        $('[onboarding-editor-tour-trigger="true"]').removeAttr('aria-controls onboarding-editor-tour-trigger');
        $trigger.attr({
            'aria-controls': popupId,
            'onboarding-editor-tour-trigger': true
        });
    }

    function showPopup(context, onNextClicked) {
        updatePopupTrigger(context);
        var $popup = createPopup.apply(this, arguments);

        // AUI 5.7 bug, setTimeout should be executed to make sure skate is ready.
        var timeout = ($("html").attr("webdriver") === "true") ? 1000 : 0; // webdriver takes 1s to ensure skate is ready
        setTimeout(function() {
            if ($popup[0].show) {
                $popup[0].show();
            } else {
                $popup[0].open = true;
            }
        }, timeout);

        return $popup;
    }

    function getCurrentPopup() {
        return document.querySelector('#' + popupId);
    }

    function onDefaultNextButtonClicked(stepName) {
        $(this).find('.next').prop('disabled', true);
        _analytics.pushEvent(EVENTS.POPUP_NEXT_BUTTON_CLICK + stepName);
        tour.next();
    }

    function removeCurrentDialog() {
        $('#' + dialogId).remove();
    }

    function removeCurrentPopup() {
        var popup = getCurrentPopup();
        popup && popup.remove();
    }

    function removeAll() {
        removeCurrentPopup();
        removeCurrentDialog();
    }

    tour.stepHandler = {
        pageTitle: function() {
            var scrollTopEl = $.browser.mozilla ? 'html' : 'body';
            $(tinyMCE.activeEditor.contentDocument).find(scrollTopEl).animate({scrollTop:0}, 'medium', function() {
                var $editorPrecursor = $("#editor-precursor");

                var $popup = showPopup({
                    alignment: 'bottom center',
                    trigger: '#content-title',
                    extraClass: 'page-title',
                    text: AJS.I18n.getText('onboarding.editor.popup.title.text'),
                    buttonClass: 'aui-button-primary next',
                    buttonText: AJS.I18n.getText('onboarding.editor.popup.button.next'),
                    stepName: "step.pageTitle"
                });
                var popupReposition = function() {
                    $popup.css({
                        'top': $editorPrecursor.offset().top + $editorPrecursor.height(),
                        'transform': 'translateX(10px)',
                        '-webkit-transform': 'translateX(10px)', /* Safari */
                        '-ms-transform': 'translateX(10px)' /* IE 9 */
                    });
                };
                /*
                 * Because editor's content will wraps into new line automatically when typing and listening events such as `scroll` or `mousewheel` can't cover this case,
                 * so ideal here is we will update popup's position for each 200 milliseconds, this interval will be destroy when we move to next step
                 */
                _interval = setInterval(popupReposition, 1000 / 30);
            });
        },
        dragDrop: function() {
            var $content = $(Confluence.Templates.OB.dialogDragDrop({data: {dialogId: dialogId}})).appendTo('body');
            $content.on('click', '.next', function() {
                $(this).prop('disabled', true);
                _analytics.pushEvent(EVENTS.DRAGDROP_NEXT_BUTTON_CLICK);
                removeCurrentDialog();
                tour.next();
            });
        },
        save: function() {
            var context = {
                alignment: 'top right',
                trigger: '#rte-button-publish',
                extraClass: 'save',
                text: AJS.I18n.getText('onboarding.editor.popup.save.text'),
                buttonClass: 'next',
                buttonText: AJS.I18n.getText('onboarding.editor.popup.button.okay'),
                stepName: "step.save"
            };
            showPopup(context, function(stepName) {
                _analytics.pushEvent(EVENTS.POPUP_NEXT_BUTTON_CLICK + stepName);
                removeAll();
                onFinishedTour();
            });

            // Add the CTA for the save button.
            $('#rte-button-publish').addClass('aui-button-primary');
        }

    };

    tour.initiate = function(analytics, onFinished) {
        _analytics = analytics;
        onFinishedTour = onFinished || onFinishedTour;
    };

    tour.setStep = function(steps) {
        _STEPS = steps;
    };

    tour.remove = function () {
        var popup = getCurrentPopup();
        popup && popup.remove();
    };

    tour.next = function() {
        removeAll();

        if (_interval) {
            clearInterval(_interval);
        }

        if (_nextStep === null) {
            _nextStep = 0;
        }

        var currentStep = _nextStep;
        _nextStep++;

        var nextFunc = tour.stepHandler[_STEPS[currentStep]];
        if (typeof nextFunc === 'function') {
            nextFunc();
        }
    };

    tour.getNextStepIndex = function () {
        return _nextStep;
    };

    tour.pause = function() {
        $('#' + popupId).hide();
        $('#' + dialogId).hide()
    };

    tour.resume = function() {
        $('#' + popupId).show();
        $('#' + dialogId).show();
    };

    return tour;
});