/**
 * Take responsibility to show blueprints option panel
 * Sequences:
 * 1. get Space Information and the list of available blueprints which are supporting by current space
 * 2. Check if selected blueprints are in available list
 * 3. Create drafts with selected blueprints
 * 4. Get drafts's content by fetching html content
 * 5. Delete all draft pages that have just created.
 * 6. Generate Option Panel.
 */
define('confluence/ob/modules/tutorial/editor/blueprints', [
    'jquery',
    'underscore',
    'confluence/ob/modules/tutorial/editor/editor',
    'confluence/ob/common/space-utils',
    'confluence/ob/common/page-utils',
    'confluence/ob/common/user'
], function ($, _, Editor, SpaceUtils, PageUtils, User) {
    'use strict';

    var _blueprintPromise = null;
    var _analytics = null;

    // Support Caching
    var _stateContent = {};
    var _stateTitle = {};
    var _stateCurrentId = null;

    var EVENTS = {
        BLUEPRINT_TEMPLATE_CLICK: "blueprint.template.click."
    };

    function saveCurrentContent(id) {
        if (id) {
            _stateContent[id] = Editor.getCurrentContent();
            _stateTitle[id] = Editor.getPageName();
        }
    }

    function restoreContent(template) {
        Editor.setCurrentContent(_stateContent[template.key] || template.content);
        Editor.changePageName(_stateTitle[template.key] || template.titleAffix);
    }

    function createOptionsPanel(templates) {
        var fullname = User.getFullName();
        var templatesData = {
            'meeting-notes-item': {
                key: 'meeting-notes-item',
                name: AJS.I18n.getText('onboarding.editor.templates.title.meeting'),
                description: AJS.I18n.getText('onboarding.editor.templates.description.meeting'),
                imageClass: 'meeting-notes',
                titleAffix: AJS.I18n.getText('onboarding.editor.blueprint.page.title.meeting', moment().format('YYYY-MM-DD')),
                content: templates['meeting-notes-item']
            },
            'requirements-item': {
                key: 'requirements-item',
                name: AJS.I18n.getText('onboarding.editor.templates.title.product'),
                description: AJS.I18n.getText('onboarding.editor.templates.description.product'),
                imageClass: 'requirements',
                titleAffix: AJS.I18n.getText('onboarding.editor.blueprint.page.title.requirement', fullname),
                content: templates['requirements-item']
            },
            'decisions-item': {
                key: 'decisions-item',
                name: AJS.I18n.getText('onboarding.editor.templates.title.decision'),
                description: AJS.I18n.getText('onboarding.editor.templates.description.decision'),
                imageClass: 'decision',
                titleAffix: AJS.I18n.getText('onboarding.editor.blueprint.page.title.decision', fullname),
                content: templates['decisions-item']
            }
        };

        var $content = $(Confluence.Templates.OB.templateOptions({options: templatesData}));

        $content.on('click', '.onboarding-editor-template-option', function (e) {
            var templateKey = $(e.currentTarget).data().optionId;
            var template = templatesData[templateKey];

            if (template) {
                _analytics.pushEvent(EVENTS.BLUEPRINT_TEMPLATE_CLICK + templateKey);

                $('.onboarding-editor-template-option').removeClass('selected');
                $(e.currentTarget).addClass('selected');

                saveCurrentContent(_stateCurrentId);
                _stateCurrentId = templateKey;

                restoreContent(template);
            }
        });

        return $content.hide().appendTo('body');
    }

    function setEditorMarginRight(value) {
        $(tinyMCE.activeEditor.dom.doc.body).css({'margin-right': value});
    }

    return {
        /**
         * Start the sequence to generate blueprint's source.
         * Since it take a few steps to complete so it nice it call this function as beginning
         */
        initiate: function (analytics) {
            _analytics = analytics;
            _blueprintPromise = Editor.getDraftTemplates(AJS.Meta.get('draft-id'));
        },

        /**
         * Create Blueprint Option Panel, This Element will be hidden as default, then you can apply any effect as you want to show it up.
         * Checking the _blueprintPromise variable before it's executed. If it was done, generate panel and resolve with jquery object as returned value,
         * otherwise execute reject method
         * @returns $Deferred
         */
        createOptionsPanel: function () {
            var deferred = $.Deferred();

            if (_blueprintPromise.state() === 'pending') {
                PageUtils.showLoadingIndicator();
            }

            _blueprintPromise.then(function (templates) {
                PageUtils.hideLoadingIndicator();

                if (!_.isEmpty(templates)) {
                    var $panel = createOptionsPanel(templates);

                    deferred.resolve($panel);
                } else {
                    deferred.reject();
                }
            }, deferred.reject.bind(deferred));

            return deferred;
        },

        /**
         * Show Blueprint Option Panel with slide animation
         * @param callback this function will be executed when the animation done
         */
        showOptionPanel: function(callback) {
            $('.onboarding-editor-template-options').fadeIn('slow', function() {
                setEditorMarginRight(450);
                if ($.isFunction(callback)) {
                    callback.call();
                }
            });
        },

        /**
         * Hide Blueprint Option Panel with slide animation
         * @param callback this function will be executed when the animation done
         */
        hideOptionPanel: function(callback) {
            $('.onboarding-editor-template-options').fadeOut('slow', function() {
                setEditorMarginRight(10);
                if ($.isFunction(callback)) {
                    callback.call();
                }
            });
        },

        isOptionPanelVisible: function() {
            return $('.onboarding-editor-template-options').is(':visible');
        },

        getSelectedTemplateId: function() {
            return _stateCurrentId;
        }
    };
});
